bl_info = {
    "name": "Ragdoll Tools",
    "category": "Rigging",
    "version": (1, 4,2),
    "blender": (2, 80,0),
    "location": "Spacebar->Ragdoll tools pie menu",
    "description": "Tools for generating ragdolls vfxmed.com ",
    "wiki_url": "https://xbodya13.github.io/ragdoll_tools_doc",
    "tracker_url": "https://github.com/xbodya13/ragdoll_tools_doc/issues"
}

import bpy
import bgl
import blf
import gpu
from gpu_extras.batch import batch_for_shader
import mathutils
# import os
import random
import math
import bmesh
import inspect
import itertools
import numpy as np
from bpy.app.handlers import persistent
import  time

ragdoll_name = "ragdoll"

bone_name = "bone"
hit_box_name = "hit_box"

location_const_name = "location_const_with_long_name"
rotation_const_name = "rotation_const_with_long_name"
scale_const_name = "scale_const_with_long_name"

const_names = {'COPY_LOCATION': location_const_name,
               'COPY_ROTATION': rotation_const_name,
               'COPY_SCALE': scale_const_name}
follow_bone_name = "follow_bone"

helpers_group_name = "ragdoll_helpers_group"
hit_box_group_name = "ragdoll_hit_boxes"
constraint_group_name="ragdoll_constraints"

default_rbc_settings = {
    "type": {"default": 'GENERIC'},
    "enabled": {"default": True},
    "disable_collisions": {"default": True},

    "use_limit_lin_x": {"default": True},
    "use_limit_lin_y": {"default": True},
    "use_limit_lin_z": {"default": True},

    "use_limit_ang_x": {"default": True},
    "use_limit_ang_y": {"default": True},
    "use_limit_ang_z": {"default": True},

    "limit_lin_x_lower": {"default": 0.0},
    "limit_lin_y_lower": {"default": 0.0},
    "limit_lin_z_lower": {"default": 0.0},

    "limit_lin_x_upper": {"default": 0.0},
    "limit_lin_y_upper": {"default": 0.0},
    "limit_lin_z_upper": {"default": 0.0},

    "limit_ang_x_lower": {"default": -0.5235987901687622},
    "limit_ang_y_lower": {"default": -0.5235987901687622},
    "limit_ang_z_lower": {"default": -0.5235987901687622},

    "limit_ang_x_upper": {"default": 0.5235987901687622},
    "limit_ang_y_upper": {"default": 0.5235987901687622},
    "limit_ang_z_upper": {"default": 0.5235987901687622},

}

default_rb_settings = {
    "type": {"default": 'ACTIVE'},
    "enabled": {"default": True},
    "collision_collections": {
        "default": [True, False, False, False, False, False, False, False, False, False, False, False, False, False,
                    False, False, False, False, False, False]},
    "use_deactivation": {"default": False},
    "collision_shape": {"default": 'CONVEX_HULL'},
    "mesh_source": {"default": 'DEFORM'},
}



class gv:
    prime_name="ragdoll_tools"

    limit_location_name=f"{prime_name}_limit_location"
    limit_rotation_name=f"{prime_name}_limit_rotation"
    limit_scale_name=f"{prime_name}_limit_scale"

    parent_holder_name=f"{prime_name}_parent_holder"

    constraint_names=(limit_location_name,limit_rotation_name,limit_scale_name,parent_holder_name)
    limit_constraint_names=(limit_location_name,limit_rotation_name,limit_scale_name)

    hit_box_bone_dictionary={}
    bone_hit_box_dictionary={}
    linked_items={}


    keymap_items=set()






def make_rigid(to_rigid, context, settings=None):
    if len(to_rigid) != 0:
        over_context = context.copy()
        over_context["object"] = to_rigid[0]
        over_context["selected_objects"] = to_rigid
        bpy.ops.rigidbody.objects_add(over_context, type='ACTIVE')
        if settings != None:
            for item in to_rigid:
                copy_settings(settings, item.rigid_body, settings.property_keys)


def get_constraint(item,name,constraint_type):
    if name not in item.constraints:
        constraint=item.constraints.new(constraint_type)
        constraint.name=name
    else:constraint=item.constraints[name]
    return constraint
def remove_constraint(item,name):
    if name in item.constraints:
        item.constraints.remove(item.constraints[name])



def set_child_of_target(constraint,target):
    if is_pose_bone(target):
        constraint.target=target.id_data
        constraint.subtarget=target.name
    else:
        constraint.target=target
        constraint.subtarget=""
def get_child_of_target(constraint):
    out=constraint.target
    if is_armature(constraint.target):
        if constraint.subtarget!="":
            out=constraint.target.pose.bones[constraint.subtarget]
    return out

def get_parent(item):
    if gv.parent_holder_name in item.constraints:
        get_child_of_target(item.constraints[gv.parent_holder_name])
def set_parent(item,parent):
    if gv.parent_holder_name in item.constraints:
        set_child_of_target(item.constraints[gv.parent_holder_name],parent)


def make_constraints(item,parent,is_head):

    limit_location=get_constraint(item,gv.limit_location_name,'LIMIT_LOCATION')
    limit_location.use_min_x=limit_location.use_min_y=limit_location.use_min_z=True
    limit_location.use_max_x=limit_location.use_max_y=limit_location.use_max_z=True

    limit_rotation=get_constraint(item,gv.limit_rotation_name,'LIMIT_ROTATION')
    limit_rotation.use_limit_x=limit_rotation.use_limit_y=limit_rotation.use_limit_z=True


    limit_scale=get_constraint(item,gv.limit_scale_name,'LIMIT_SCALE')
    limit_scale.use_min_x=limit_scale.use_min_y=limit_scale.use_min_z=True
    limit_scale.use_max_x=limit_scale.use_max_y=limit_scale.use_max_z=True

    matrix_world=item.matrix_world

    location,quaternion,scale=matrix_world.decompose()
    rotation=quaternion.to_euler()
    if item.rotation_mode not in ('QUATERNION','AXIS_ANGLE'):
        rotation=item.rotation_euler.copy()
        rotation.zero()
        rotation.rotate(quaternion)

    limit_location.min_x=limit_location.max_x=location.x
    limit_location.min_y=limit_location.max_y=location.y
    limit_location.min_z=limit_location.max_z=location.z

    limit_rotation.min_x=limit_rotation.max_x=rotation.x
    limit_rotation.min_y=limit_rotation.max_y=rotation.y
    limit_rotation.min_z=limit_rotation.max_z=rotation.z

    limit_scale.min_x=limit_scale.max_x=scale.x
    limit_scale.min_y=limit_scale.max_y=scale.y
    limit_scale.min_z=limit_scale.max_z=scale.z

    item.ragdoll_tools_own_rest_matrix=flatten(item.matrix_world)
    item.ragdoll_tools_other_rest_matrix=flatten(parent.matrix_world)

    parent.ragdoll_tools_own_rest_matrix=flatten(parent.matrix_world)
    parent.ragdoll_tools_other_rest_matrix=flatten(item.matrix_world)



    if is_head:
        limit_location.mute=True
        limit_rotation.mute=True
        limit_scale.mute=True

    else:
        parent_holder=get_constraint(item,gv.parent_holder_name,'CHILD_OF')
        set_child_of_target(parent_holder,parent)


    if is_pose_bone(item):
        limit_location.owner_space='POSE'
        limit_rotation.owner_space='POSE'
        limit_scale.owner_space='POSE'
        if not is_head:
            # parent_holder.inverse_matrix=(item.id_data.matrix_world.inverted()@parent.matrix_world).inverted()
            parent_holder.inverse_matrix=parent.ragdoll_tools_own_rest_matrix.inverted()
    else:
        if not is_head:
            # parent_holder.inverse_matrix=parent.matrix_world.inverted_safe()
            parent_holder.inverse_matrix=parent.ragdoll_tools_own_rest_matrix.inverted()


def has_constraints(item):
    for name in  gv.constraint_names:
        if name not in item.constraints:
            return False
    return True

def has_limit_constraints(item):
    for name in  gv.limit_constraint_names:
        if name not in item.constraints:
            return False
    return True

def clear_constraints(item):
    for name in gv.constraint_names:
        if name in item.constraints:
            item.constraints.remove(item.constraints[name])


def get_constraints_states(item):
    out=[]
    for name in gv.constraint_names:
        if name in item.constraints:
            out.append(not item.constraints[name].mute)
        else:out.append(False)
    return out

def set_constraints_states(item,states):
    for name,state in zip(gv.constraint_names,states):
        if name in item.constraints:
            item.constraints[name].mute=not state



def set_constraints_state(item,state=True):
    for name in gv.constraint_names:
        if name in item.constraints:
            item.constraints[name].mute=not state





def link_hit_box_and_bone(hit_box,pose_bone,context):

    make_constraints(hit_box,pose_bone,True)
    make_constraints(pose_bone,hit_box,False)







def update_pairs():

    pairs=set()


    for source_object in bpy.data.objects:
        if is_armature(source_object):
            for pose_bone in source_object.pose.bones:
                if has_constraints(pose_bone):
                    hit_box=get_child_of_target(pose_bone.constraints[gv.parent_holder_name])
                    if is_mesh(hit_box):
                        if has_limit_constraints(hit_box):
                            pairs.add((hit_box,pose_bone))

        if is_mesh(source_object):
            if has_constraints(source_object):
                pose_bone=get_child_of_target(source_object.constraints[gv.parent_holder_name])
                if is_pose_bone(pose_bone):
                    if has_limit_constraints(pose_bone):
                        pairs.add((source_object,pose_bone))



    gv.hit_box_bone_dictionary={}
    gv.bone_hit_box_dictionary={}
    gv.linked_items={}
    for hit_box,hit_bone in pairs:
        gv.hit_box_bone_dictionary[hit_box]=hit_bone
        gv.bone_hit_box_dictionary[hit_bone]=hit_box

        gv.linked_items[hit_box]=hit_bone
        gv.linked_items[hit_bone]=hit_box







def get_pair(item):
    hit_box = None
    hit_bone = None

    if item in gv.linked_items:
        if is_mesh(item):
            hit_box=item
            hit_bone=gv.hit_box_bone_dictionary[hit_box]
        if type(item)==bpy.types.PoseBone:
            hit_bone=item
            hit_box=gv.bone_hit_box_dictionary[hit_bone]

    return hit_box, hit_bone

def get_other(item):
    return gv.linked_items[item]

def parent_rest_matrix(item):
    other_item=gv.linked_items[item]


    return other_item.matrix_world@item.ragdoll_tools_other_rest_matrix.inverted()@item.ragdoll_tools_own_rest_matrix

def set_pair_head(item):
    if item in gv.linked_items:
        head=item
        head_matrix=head.matrix_world.copy()
        tail=gv.linked_items[item]
        tail_matrix=tail.matrix_world.copy()

        set_parent(head,None)
        remove_constraint(head,gv.parent_holder_name)
        set_constraints_state(head,False)

        constraint=get_constraint(tail,gv.parent_holder_name,'CHILD_OF')
        constraint.inverse_matrix=head.ragdoll_tools_own_rest_matrix.inverted()
        set_parent(tail,head)
        set_constraints_state(tail,True)

        head.matrix_world=head_matrix
        tail.matrix_world=parent_rest_matrix(tail)


def flatten(m):
    out=()
    for col in m.col:
        out+=tuple(col)
    return out

def random_matrix():
    return mathutils.Matrix.Translation([random.randint(0, 5), random.randint(0, 5), random.randint(0, 5)])

def compose_matrix(l=mathutils.Vector((0,0,0)),r=mathutils.Quaternion((1,0,0,0)),s=mathutils.Vector((1,1,1))):
    # print(s)
    scale_matrix=mathutils.Matrix()
    scale_matrix[0][0] = s[0]
    scale_matrix[1][1] = s[1]
    scale_matrix[2][2] = s[2]
    return mathutils.Matrix.Translation(l) @ mathutils.Matrix.Rotation(r.angle, 4, r.axis)@scale_matrix

def get_center_of_mass(mesh):
    # print(sum([v.co for v in mesh.vertices],mathutils.Vector((0,0,0))))
    out = mathutils.Vector((0, 0, 0))
    if len(mesh.vertices)!=0:
        out=sum([ v.co for v  in mesh.vertices],out)/len(mesh.vertices)
    return out

def set_origin(some_object,matrix):
    shift_matrix=some_object.matrix_world.inverted()@matrix
    some_object.matrix_world@=shift_matrix
    some_object.data.transform(shift_matrix.inverted())

    # some_object.matrix_world=mathutils.Matrix.Identity(4)


class KeyframerInsert(bpy.types.Operator):
    """Insert keyframe on selected hitboxes"""
    bl_idname="ragdoll.keyframer_insert"
    bl_label=""
    bl_options=set()

    def execute(self,context):
        bpy.ops.ragdoll.keyframer(mode='INSERT')
        return {'FINISHED'}

class KeyframerDelete(bpy.types.Operator):
    """Delete keyframe at a current frame"""
    bl_idname="ragdoll.keyframer_delete"
    bl_label=""
    bl_options=set()

    def execute(self,context):
        bpy.ops.ragdoll.keyframer(mode='DELETE')
        return {'FINISHED'}

class KeyframerClear(bpy.types.Operator):
    """Clear keyframes"""
    bl_idname="ragdoll.keyframer_clear"
    bl_label=""
    bl_options=set()

    def execute(self,context):
        bpy.ops.ragdoll.keyframer(mode='CLEAR')
        return {'FINISHED'}

class KeyframerEnable(bpy.types.Operator):
    """Make selected hitboxes animated"""
    bl_idname="ragdoll.keyframer_enable"
    bl_label="Make animated"
    bl_options=set()

    def execute(self,context):
        bpy.ops.ragdoll.keyframer(mode='ENABLE')
        return {'FINISHED'}

class KeyframerDisable(bpy.types.Operator):
    """Make selected hitboxes dynamic"""
    bl_idname="ragdoll.keyframer_disable"
    bl_label="Make dynamic"
    bl_options=set()

    def execute(self,context):
        bpy.ops.ragdoll.keyframer(mode='DISABLE')
        return {'FINISHED'}


class Keyframer(bpy.types.Operator):
    """Insert or delete keyframes"""
    bl_idname="ragdoll.keyframer"
    bl_label="Keyframer"
    bl_options={'REGISTER','UNDO'}

    mode: bpy.props.EnumProperty(name="Mode",
                                 items=[
                                     ('INSERT',"Insert","Insert",'NONE',0),
                                     ('DELETE',"Delete","Delete",'NONE',1),
                                     ('CLEAR',"Clear","Clear",'NONE',2),
                                     ('ENABLE',"Enable","Enable",'NONE',3),
                                     ('DISABLE',"Disable","Disable",'NONE',4),
                                 ]
                                 )


    @classmethod
    def poll(self,context):
        return True

    def execute(self,context):


        if context.mode=='POSE':
            selected_items=[gv.bone_hit_box_dictionary[item] for item in context.selected_pose_bones if item in gv.bone_hit_box_dictionary]
        if context.mode=='OBJECT':
            selected_items=context.selected_objects

        selected_items=[item for item in selected_items if item.rigid_body is not None]

        if self.mode in ('ENABLE','DISABLE'):
            for item in selected_items:
                if self.mode=='ENABLE':item.rigid_body.kinematic=True
                if self.mode=='DISABLE':item.rigid_body.kinematic=False


        for item in selected_items:

            data_path="rigid_body.kinematic"
            base_path=""
            fcurve_holder=item

            if data_path is not None:
                if self.mode=='INSERT':
                    item.keyframe_insert(data_path)
                if self.mode=='DELETE':
                    if fcurve_holder.animation_data is not None:
                        if fcurve_holder.animation_data.action is not None:
                            item.keyframe_delete(data_path)
                if self.mode=='CLEAR':
                    if fcurve_holder.animation_data is not None:
                        if fcurve_holder.animation_data.action is not None:
                            to_remove=fcurve_holder.animation_data.action.fcurves.find(base_path+data_path)
                            if to_remove is not None:
                                fcurve_holder.animation_data.action.fcurves.remove(to_remove)
                                fcurve_holder.animation_data.action.fcurves.update()

        bpy.ops.wm.redraw_timer(type='DRAW_WIN_SWAP',iterations=1)

        return {'FINISHED'}





class BakeToBone(bpy.types.Operator):
    """Bake connected hitbox or bones animation to keyframes """
    bl_idname = "ragdoll.bake_to_bone"
    bl_label = "Bake"
    bl_options = {'REGISTER', 'UNDO'}

    bake_mode : bpy.props.EnumProperty(name="Bake mode",
                                       items=[
                                           ('BONES', "Bones animation bake", "Bakes bones animation to keyframes",
                                            'NONE', 0),
                                           ('BOXES', "Hitboxes animation bake", "Bakes hitbox animation to keyframes",
                                            'NONE', 1),
                                           ('BOTH', "Both", "Bakes hitbox and bones animation to keyframes", 'NONE', 2)
                                       ]
                                       )
    use_selected_bones_only : bpy.props.BoolProperty(name="Use selected bones only")
    frame_start : bpy.props.IntProperty(
        name="Start Frame",
        description="Start frame for baking",
        min=0, max=300000,
        default=1,
    )
    frame_end : bpy.props.IntProperty(
        name="End Frame",
        description="End frame for baking",
        min=1, max=300000,
        default=250,
    )

    def invoke(self, context, event):

        return context.window_manager.invoke_props_dialog(self)

    def execute(self, context):
        update_pairs()

        frame_range = range(self.frame_start, self.frame_end)

        hit_bone_set = set()
        hit_box_set = set()

        def add_to_set(item):
            hit_box, hit_bone = get_pair(item)
            if None not in (hit_box, hit_bone):
                hit_box_set.add(hit_box)
                hit_bone_set.add(hit_bone)

        if context.mode == 'POSE':
            if self.use_selected_bones_only:
                selected_pose_bones = context.selected_pose_bones
            else:
                selected_pose_bones = context.visible_pose_bones
            for pose_bone in selected_pose_bones:
                add_to_set(pose_bone)


        else:
            for selected_object in context.selected_objects:
                if context.mode != 'POSE':
                    if is_armature(selected_object):
                        for pose_bone in selected_object.pose.bones:
                            if not pose_bone.bone.hide:
                                if pose_bone.bone.select or not self.use_selected_bones_only:
                                    if any([bl and al for bl, al in
                                            zip(pose_bone.bone.layers, selected_object.data.layers)]):
                                        add_to_set(pose_bone)

                if is_mesh(selected_object):
                    add_to_set(selected_object)

        if self.bake_mode == 'BONES':
            bake_items = hit_bone_set
        if self.bake_mode == 'BOXES':
            bake_items = hit_box_set
        if self.bake_mode == 'BOTH':
            bake_items = hit_box_set.union(hit_bone_set)



        if len(bake_items) != 0:
            last_frame = context.scene.frame_current

            trajectories = [[] for item in bake_items]
            for f in frame_range:
                # context.scene.frame_set(f)
                # context.scene.frame_set(f)
                context.scene.frame_set(f)
                # context.scene.frame_set(f)
                # context.scene.update()
                # context.scene.frame_current=f

                for trajectory, item in zip(trajectories, bake_items):
                    if type(item) == bpy.types.PoseBone:
                        # item.matrix=item.matrix*mathutils.Matrix.Identity(4)
                        trajectory.append((item.id_data.convert_space(pose_bone=item, matrix=item.matrix, from_space='POSE',to_space= 'LOCAL'), f))
                    if type(item) == bpy.types.Object:
                        # item.matrix_world = item.matrix_world * mathutils.Matrix.Identity(4)
                        trajectory.append((item.matrix_world.copy(), f))

            options = {'INSERTKEY_NEEDED'}
            # options=set()
            # options = {'INSERTKEY_VISUAL'}

            for item, trajectory in zip(bake_items, trajectories):
                euler_prev = None
                # print(trajectory[0])
                for matrix, f in trajectory:
                    item.matrix_basis = matrix.copy()

                    item.keyframe_insert("location", -1, f, item.name, options)

                    rotation_mode = item.rotation_mode
                    if rotation_mode == 'QUATERNION':
                        item.keyframe_insert("rotation_quaternion", -1, f, item.name, options)
                    elif rotation_mode == 'AXIS_ANGLE':
                        item.keyframe_insert("rotation_axis_angle", -1, f, item.name, options)
                    else:  # euler, XYZ, ZXY etc
                        if euler_prev is not None:
                            euler = item.rotation_euler.copy()
                            euler.make_compatible(euler_prev)
                            item.rotation_euler = euler
                            euler_prev = euler
                            del euler
                        else:
                            euler_prev = item.rotation_euler.copy()
                        item.keyframe_insert("rotation_euler", -1, f, item.name, options)

                    item.keyframe_insert("scale", -1, f, item.name, options)

            context.scene.frame_set(last_frame)
            context.scene.frame_set(last_frame)
            # context.view_layer.update()

        return {'FINISHED'}


class ClearCache(bpy.types.Operator):
    """Clear rigid body cache"""
    bl_idname = "ragdoll.clear_cache"
    bl_label = "Clear cache"
    bl_options = {'REGISTER', 'UNDO'}


    # def draw(self, context):
    #     layout = self.layout

    # @classmethod
    # def poll(self, context):
    #     return context.mode in ('OBJECT','POSE')


    def execute(self, context):

        if context.scene.rigidbody_world is not None:

            world_names=['collection','constraints','enabled','time_scale','steps_per_second','solver_iterations','use_split_impulse','substeps_per_frame']

            effector_names=['apply_to_hair_growing','collection','gravity','all','force','vortex','magnetic','wind','curve_guide','texture','harmonic','charge','lennardjones','boid','turbulence','drag','smokeflow']


            point_cache_names=[ 'frame_start', 'frame_end']

            world_properties={name: getattr(context.scene.rigidbody_world,name) for name in world_names if hasattr(context.scene.rigidbody_world,name)}
            effector_properties={name:getattr(context.scene.rigidbody_world.effector_weights,name) for name in effector_names}
            point_cache_properties={name:getattr(context.scene.rigidbody_world.point_cache,name) for name in point_cache_names}


            context.scene.rigidbody_world.collection=None
            context.scene.rigidbody_world.constraints=None



            bpy.ops.rigidbody.world_remove()
            bpy.ops.rigidbody.world_add()



            for name in world_names:
                if hasattr(context.scene.rigidbody_world,name):
                    setattr(context.scene.rigidbody_world,name,world_properties[name])

            for name in effector_names:
                setattr(context.scene.rigidbody_world.effector_weights,name,effector_properties[name])


            for name in point_cache_names:
                setattr(context.scene.rigidbody_world.point_cache,name,point_cache_properties[name])



            tmp=bpy.data.objects.new("",None)
            context.scene.collection.objects.link(tmp)
            bpy.data.objects.remove(tmp,do_unlink=True)


            # context.view_layer.update()
            # bpy.ops.wm.redraw_timer(type='DRAW_WIN_SWAP',iterations=1)


        return {'FINISHED'}






def register_extensions():

    # setattr(bpy.types.Object, follow_bone_name, bpy.props.BoolProperty(name="Follow bone", default=True))


    bpy.types.Object.parent_to = make_parent
    bpy.types.Object.location_world = property(object_location_world)
    bpy.types.Object.center_world = property(object_location_world)

    bpy.types.Object.ragdoll_tools_own_rest_matrix=bpy.props.FloatVectorProperty(size=16,subtype='MATRIX')
    bpy.types.Object.ragdoll_tools_other_rest_matrix=bpy.props.FloatVectorProperty(size=16,subtype='MATRIX')




    bpy.types.PoseBone.matrix_world = property(pose_bone_world_get,pose_bone_world_set)
    bpy.types.PoseBone.matrix_world_rest = property(pose_bone_world_rest_matrix)

    bpy.types.PoseBone.head_world = property(pose_bone_head_world)
    bpy.types.PoseBone.tail_world = property(pose_bone_tail_world)
    bpy.types.PoseBone.center_world = property(pose_bone_center_world)

    bpy.types.PoseBone.head_world_rest = property(pose_bone_head_world_rest)
    bpy.types.PoseBone.tail_world_rest = property(pose_bone_tail_world_rest)
    bpy.types.PoseBone.center_world_rest = property(pose_bone_center_world_rest)


    bpy.types.PoseBone.ragdoll_tools_own_rest_matrix=bpy.props.FloatVectorProperty(size=16,subtype='MATRIX')
    bpy.types.PoseBone.ragdoll_tools_other_rest_matrix=bpy.props.FloatVectorProperty(size=16,subtype='MATRIX')




def make_parent(child_item, parent_item, use_rest=False):
    if type(parent_item) == bpy.types.PoseBone:
        child_item.parent = parent_item.id_data
        child_item.parent_type = 'BONE'
        child_item.parent_bone = parent_item.name
        if use_rest:
            child_item.matrix_parent_inverse = (
                        parent_item.id_data.matrix_world @ parent_item.bone.matrix_local @ mathutils.Matrix.Translation(
                    mathutils.Vector([0, parent_item.bone.length, 0]))).inverted()
        else:
            child_item.matrix_parent_inverse = (
                        parent_item.id_data.matrix_world @ parent_item.matrix @ mathutils.Matrix.Translation(
                    mathutils.Vector([0, parent_item.bone.length, 0]))).inverted()
    if type(parent_item) == bpy.types.Object:
        child_item.parent = parent_item
        child_item.matrix_parent_inverse = parent_item.matrix_world.inverted()


def object_location_world(some_object):
    return some_object.matrix_world.to_translation()


def pose_bone_world_get(pose_bone):
    return pose_bone.id_data.matrix_world @ pose_bone.matrix
    # return pose_bone.id_data.matrix_world @pose_bone.id_data.matrix_world @ pose_bone.matrix
    # return pose_bone.id_data.convert_space(pose_bone=pose_bone,matrix=pose_bone.matrix,from_space='POSE',to_space='WORLD')
def pose_bone_world_set(pose_bone,matrix):
    pose_bone.matrix=pose_bone.id_data.matrix_world.inverted()@matrix

def pose_bone_world_rest_matrix(pose_bone):
    return pose_bone.id_data.matrix_world @ pose_bone.bone.matrix_local


def pose_bone_head_world(pose_bone):
    return pose_bone.id_data.matrix_world @ pose_bone.head


def pose_bone_tail_world(pose_bone):
    return pose_bone.id_data.matrix_world @ pose_bone.tail


def pose_bone_center_world(pose_bone):
    return pose_bone.id_data.matrix_world @ pose_bone.center


def pose_bone_head_world_rest(pose_bone):
    return pose_bone.id_data.matrix_world @ pose_bone.bone.head_local


def pose_bone_tail_world_rest(pose_bone):
    return pose_bone.id_data.matrix_world @ pose_bone.bone.tail_local


def pose_bone_center_world_rest(pose_bone):
    return (
                       pose_bone.id_data.matrix_world @ pose_bone.bone.head_local + pose_bone.id_data.matrix_world @ pose_bone.bone.tail_local) * 0.5


def print_context(context):
    work_context = context.copy()
    print("\n\n|||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||\n\n")
    for key in sorted(work_context.keys()):
        print("->", key, "   ", work_context[key])


def print_dir(item):
    print("\n\n|||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||\n\n")
    for key in sorted(dir(item)):
        print("->", key, "   ", str(getattr(item, key)))


def prop_from_type(rna_type, **override):
    out = None

    if type(rna_type) == bpy.types.EnumProperty:
        out = bpy.props.EnumProperty(
            items=[(item.identifier, item.name, item.description, item.icon, x) for x, item in
                   enumerate(rna_type.enum_items)],
            name=rna_type.name,
            description=rna_type.description,
            default=rna_type.default
        )
    if type(rna_type) == bpy.types.BoolProperty:
        if rna_type.is_array:
            subtype_mapping = {'LAYER_MEMBERSHIP': 'LAYER','LAYER_MEMBER': 'LAYER'}


            out = bpy.props.BoolVectorProperty(
                name=rna_type.name,
                description=rna_type.description,
                subtype=subtype_mapping[rna_type.subtype],
                size=rna_type.array_length,
                default=rna_type.default_array

            )

        else:
            out = bpy.props.BoolProperty(
                name=rna_type.name,
                description=rna_type.description,
                default=rna_type.default,
                subtype=rna_type.subtype
            )
    if type(rna_type) == bpy.types.FloatProperty:
        subtype_mapping={'':'NONE'}
        out = bpy.props.FloatProperty(
            name=rna_type.name,
            description=rna_type.description,
            default=rna_type.default,
            # subtype=subtype_mapping[rna_type.subtype],
            unit=rna_type.unit,

            min=rna_type.hard_min,
            max=rna_type.hard_max,
            soft_min=rna_type.soft_min,
            soft_max=rna_type.soft_max,

            step=rna_type.step,
            precision=rna_type.precision

        )
    if type(rna_type) == bpy.types.IntProperty:
        out = bpy.props.IntProperty(
            name=rna_type.name,
            description=rna_type.description,
            default=rna_type.default,
            subtype=rna_type.subtype,

            min=rna_type.hard_min,
            max=rna_type.hard_max,
            soft_min=rna_type.soft_min,
            soft_max=rna_type.soft_max,

            step=rna_type.step,

        )
    for key, value in override.items():
        if key in out[1].keys():
            out[1][key] = value
    return out


def pgroup_from_type(b_type, **override):
    def pgroup_from_type_inner(c_object):
        property_keys = set()
        if not hasattr(c_object,'__annotations__'): c_object.__annotations__={}
        for p_name, p_type in b_type.bl_rna.properties.items():
            if prop_from_type(p_type) != None:
                # print(type(c_object))
                if p_name in override.keys():
                    # setattr(c_object, p_name, prop_from_type(p_type, **override[p_name]))
                    c_object.__annotations__[p_name]=prop_from_type(p_type, **override[p_name])
                else:
                    # setattr(c_object, p_name, prop_from_type(p_type))
                    c_object.__annotations__[p_name]=prop_from_type(p_type)

                property_keys.add(p_name)

        setattr(c_object, "property_keys", property_keys)
        return c_object

    return pgroup_from_type_inner


def inner_pgroups_to_pointers(c_object):
    pgroups = []

    if not hasattr(c_object,'__annotations__'):c_object.__annotations__={}
    for pgroup_name in vars(c_object):
        tmp = getattr(c_object, pgroup_name)
        if inspect.isclass(tmp):
            if issubclass(tmp, bpy.types.PropertyGroup):
                # print(c_object,tmp)
                bpy.utils.register_class(tmp)
                pgroups.append(pgroup_name)
    for pgroup_name in pgroups:
        tmp = getattr(c_object, pgroup_name)
        d_name = pgroup_name + "_definition"
        setattr(c_object, d_name, tmp)
        # setattr(c_object, pgroup_name, bpy.props.PointerProperty(type=getattr(c_object, d_name))  )
        c_object.__annotations__[pgroup_name]=bpy.props.PointerProperty(type=getattr(c_object, d_name))
        # bpy.utils.register_class(getattr(c_object, d_name))

    return c_object


def is_armature(t_object):
    return t_object != None and type(t_object) == bpy.types.Object and type(t_object.data) == bpy.types.Armature


def is_mesh(t_object):
    return t_object != None and type(t_object) == bpy.types.Object and type(t_object.data) == bpy.types.Mesh

def is_pose_bone(t_object):
    return t_object != None and type(t_object) == bpy.types.PoseBone


def copy_settings(from_class, to_class, keys):
    for key in keys:
        if hasattr(from_class, key) and hasattr(to_class, key):
            setattr(to_class, key, getattr(from_class, key))




def length(a, b):
    return (a - b).length


def get_nearest(some_object, targets):
    min_l = 0
    min_object = None
    for target in targets:

        l = length(some_object.center_world, target.center_world)
        if min_object == None or l < min_l:
            min_l = l
            min_object = target
    return min_object

def get_nearest_mapping(hit_boxes,bones):

    if len(bones)<len(hit_boxes):
        short=bones
        long=hit_boxes
    else:
        short=hit_boxes
        long=bones

    short_size=len(short)
    long_size=len(long)


    weights=np.zeros((short_size,long_size))

    for x in range(short_size):
        for y in range(long_size):
            weights[x][y]=length(short[x].center_world, long[y].center_world)

    out=[]
    for x in range(short_size):
        short_item=short[x]
        long_item=long[np.argmin(weights[x])]

        if is_mesh(short_item):
            out.append((short_item,long_item))
        else:
            out.append((long_item,short_item))

    return list(set(out))





@inner_pgroups_to_pointers
class LinkHitBoxAndBone(bpy.types.Operator):
    """Connect selected objects to the nearest bones in selected armatures"""
    bl_idname = "ragdoll.link_hit_boxes_and_bones"
    bl_label = "Connect"
    bl_options = {'REGISTER', 'UNDO'}

    use_selected_bones_only : bpy.props.BoolProperty(name="Use selected bones only")



    def check(self, context):
        return True

    def draw(self, context):
        layout = self.layout

        layout.prop(self, "use_selected_bones_only")


    def execute(self, context):
        update_pairs()
        mesh_list = []
        pose_bone_list = []

        if context.mode == 'POSE':
            if self.use_selected_bones_only:
                selected_pose_bones = context.selected_pose_bones
            else:
                selected_pose_bones = context.visible_pose_bones
            for pose_bone in selected_pose_bones:
                if None in get_pair(pose_bone):
                    pose_bone_list.append(pose_bone)

        for selected_object in context.selected_objects:
            if context.mode != 'POSE':
                if is_armature(selected_object):
                    for pose_bone in selected_object.pose.bones:
                        if not pose_bone.bone.hide:
                            if pose_bone.bone.select or not self.use_selected_bones_only:
                                if any([bl and al for bl, al in
                                        zip(pose_bone.bone.layers, selected_object.data.layers)]):
                                    if None in get_pair(pose_bone):
                                        pose_bone_list.append(pose_bone)

            if is_mesh(selected_object):
                if None in get_pair(selected_object):
                    mesh_list.append(selected_object)

        remove_rig(pose_bone_list)
        remove_rig(mesh_list)


        # if len(mesh_list) <= len(pose_bone_list):
        #     tmp_set = set(pose_bone_list)
        #     for mesh in mesh_list:
        #         nearest_pose_bone = get_nearest(mesh, tmp_set)
        #         link_hit_box_and_bone(mesh,nearest_pose_bone,context)
        #         tmp_set.remove(nearest_pose_bone)
        # else:
        #     tmp_set = set(mesh_list)
        #     for pose_bone in pose_bone_list:
        #         nearest_mesh = get_nearest(pose_bone, tmp_set)
        #         link_hit_box_and_bone(nearest_mesh,pose_bone,context)
        #         tmp_set.remove(nearest_mesh)


        for mesh,pose_bone in get_nearest_mapping(mesh_list,pose_bone_list):
            # print(mesh,pose_bone)
            link_hit_box_and_bone(mesh,pose_bone,context)

        to_rigid=[mesh for mesh in mesh_list if mesh.rigid_body is None]
        if len(to_rigid)!=0:
            make_rigid(to_rigid,context)


        return {'FINISHED'}


def remove_rig(items):
    names=("ragdoll_tools_own_rest_matrix","ragdoll_tools_other_rest_matrix")

    for item in items:
        constraints_to_delete = set()
        for constraint in item.constraints:
            if constraint.name in gv.constraint_names:
                constraints_to_delete.add(constraint)
        for constraint in constraints_to_delete:
            item.constraints.remove(constraint)


        for name in names:
            if name in item:
                del item[name]




class SetMass(bpy.types.Operator):
    """Set total mass of selected objects"""
    bl_idname = "ragdoll.set_mass"
    bl_label = "Set mass"
    bl_options = {'REGISTER', 'UNDO'}

    mass:bpy.props.FloatProperty(name="Total mass",description="Total mass of selected objects",min=0, default=80,unit='MASS')


    # def draw(self, context):
    #     layout = self.layout

    @classmethod
    def poll(self, context):
        return context.mode in ('OBJECT','POSE')


    def execute(self, context):
        update_pairs()

        if context.mode=='POSE':
            hit_boxes=[gv.bone_hit_box_dictionary[item] for item in context.selected_pose_bones if item in gv.bone_hit_box_dictionary]
        if context.mode=='OBJECT':
            hit_boxes=context.selected_objects

        hit_boxes=[item for item in hit_boxes if item.rigid_body is not None]

        if len(hit_boxes)!=0:


            last_mode=context.mode
            last_selected=context.selected_objects.copy()
            last_active=context.view_layer.objects.active

            context.view_layer.objects.active=hit_boxes[0]
            bpy.ops.object.mode_set(mode='OBJECT')


            for item in bpy.data.objects:item.select_set(False)
            for hit_box in hit_boxes:hit_box.select_set(True)

            bpy.ops.rigidbody.mass_calculate()
            current_mass=sum(hit_box.rigid_body.mass for hit_box in hit_boxes)

            if current_mass!=0:
                for hit_box in hit_boxes:
                    hit_box.rigid_body.mass*=self.mass/current_mass



            for item in bpy.data.objects: item.select_set(False)
            for item in last_selected: item.select_set(True)

            context.view_layer.objects.active=last_active
            bpy.ops.object.mode_set(mode=last_mode)



        return {'FINISHED'}


class SetLimits(bpy.types.Operator):
    """Interactively set constraint limits"""
    bl_idname = "ragdoll.set_limits"
    bl_label = "Set limits"
    bl_options = {'REGISTER', 'UNDO'}

    mode : bpy.props.EnumProperty(name="Settings",options={'HIDDEN'},
                                          items=[
                                              ('START', "", "", 'NONE', 0),
                                              ('MODE',"","",'NONE',1),
                                              ('CLEAR',"","",'NONE',2),
                                              ('HELP',"","",'NONE',3),
                                              ('STOP',"","",'NONE',4),

                                          ]
                                          )
    update_mode=0
    show_help=True

    class ConstraintHolder:
        def __init__(self,constraint):
            self.name=constraint.name

            self.is_valid=False

            self.constraint=None
            self.node_a=None
            self.node_b=None



            self.constraint_initial_matrix=None
            self.initial_matrix_a=None
            self.initial_matrix_b=None

            self.work_euler=None
            self.last_quaternion=None

            self.visual_matrix=None

            self.validate()

        def validate(self):
            self.node_a=None
            self.node_b=None
            if self.name in bpy.data.objects:
                self.constraint=bpy.data.objects[self.name]
                if self.constraint.rigid_body_constraint is not None:
                    rbc=self.constraint.rigid_body_constraint
                    self.node_a=rbc.object1
                    self.node_b=rbc.object2
            if None in (self.node_a,self.node_b):self.is_valid=False
            else:self.is_valid=True



        def update_initial(self):
            self.constraint_initial_matrix=self.constraint.matrix_world.copy()
            self.initial_matrix_a=self.node_a.matrix_world.copy()
            self.initial_matrix_b=self.node_b.matrix_world.copy()

            self.work_euler=None
            self.last_quaternion=None



        def update_limits(self):

            matrix_a=self.node_a.matrix_world@self.initial_matrix_a.inverted()@self.constraint_initial_matrix
            matrix_b=self.node_b.matrix_world@self.initial_matrix_b.inverted()@self.constraint_initial_matrix


            # mode=bpy.context.scene.ragdoll_tools_limits_mode
            mode=SetLimits.update_mode

            if mode in (0,1):
                location=matrix_a.inverted()@matrix_b.to_translation()
                current_location_min,current_location_max=self.get_limits(location_min=True,location_max=True)

                location_min=mathutils.Vector(min(location[x],current_location_min[x]) for x in range(3))
                location_max=mathutils.Vector(max(location[x],current_location_max[x]) for x in range(3))

                self.set_limits(location_min=location_min,location_max=location_max)

            if mode in (0,2):
                quaternion=matrix_b.to_quaternion().inverted()@matrix_a.to_quaternion()



                if self.last_quaternion is None:
                    self.work_euler=quaternion.to_euler()
                    self.last_quaternion=quaternion
                else:

                    self.work_euler.rotate(self.last_quaternion.inverted()@quaternion)
                    self.last_quaternion=quaternion




                current_rotation_min,current_rotation_max=self.get_limits(rotation_min=True,rotation_max=True)

                rotation_min=mathutils.Vector(min(self.work_euler[x],current_rotation_min[x]) for x in range(3))
                rotation_max=mathutils.Vector(max(self.work_euler[x],current_rotation_max[x]) for x in range(3))

                self.set_limits(rotation_min=rotation_min,rotation_max=rotation_max)


        def reset_constraint(self):
            limit=mathutils.Vector()
            # mode=bpy.context.scene.ragdoll_tools_limits_mode
            mode=SetLimits.update_mode
            if mode==0: self.set_limits(limit,limit,limit,limit)
            if mode==1: self.set_limits(location_min=limit,location_max=limit)
            if mode==2: self.set_limits(rotation_min=limit,rotation_max=limit)
            self.update_initial()

        def get_limits(self,location_min=False,location_max=False,rotation_min=False,rotation_max=False):
            rbc=self.constraint.rigid_body_constraint
            out=[]
            if location_min:
                out.append(mathutils.Vector((rbc.limit_lin_x_lower,rbc.limit_lin_y_lower,rbc.limit_lin_z_lower)))
            if location_max:
                out.append(mathutils.Vector((rbc.limit_lin_x_upper,rbc.limit_lin_y_upper,rbc.limit_lin_z_upper)))

            if rotation_min:
                out.append(mathutils.Vector((rbc.limit_ang_x_lower,rbc.limit_ang_y_lower,rbc.limit_ang_z_lower)))
            if rotation_max:
                out.append(mathutils.Vector((rbc.limit_ang_x_upper,rbc.limit_ang_y_upper,rbc.limit_ang_z_upper)))
            return out


        def set_limits(self,location_min=None,location_max=None,rotation_min=None,rotation_max=None):
            rbc=self.constraint.rigid_body_constraint

            if location_min is not None:
                rbc.limit_lin_x_lower,rbc.limit_lin_y_lower,rbc.limit_lin_z_lower=location_min
            if location_max is not None:
                rbc.limit_lin_x_upper,rbc.limit_lin_y_upper,rbc.limit_lin_z_upper=location_max

            if rotation_min is not None:
                rbc.limit_ang_x_lower,rbc.limit_ang_y_lower,rbc.limit_ang_z_lower=rotation_min
            if rotation_max is not None:
                rbc.limit_ang_x_upper,rbc.limit_ang_y_upper,rbc.limit_ang_z_upper=rotation_max

    keymap_items=set()

    holders=[]
    valid_holders=[]

    text_handler=None
    limits_handler=None
    shader=gpu.shader.from_builtin('3D_SMOOTH_COLOR')
    shader.bind()



    @staticmethod
    def update_initial():
        SetLimits.holders=[]
        for source_object in bpy.data.objects:
            holder=SetLimits.ConstraintHolder(source_object)
            if holder.is_valid:
                holder.update_initial()
                SetLimits.holders.append(holder)



    @staticmethod
    def handler(scene):
        # print("SET LIMITS HANDLER",random.randint(0,100))
        for holder in SetLimits.holders:holder.validate()
        for holder in SetLimits.holders:
            if holder.is_valid:holder.update_limits()


    @staticmethod
    def text_handler_source(self,context):


        # print(bpy.context.region)

        if SetLimits.show_help:
            # mode=bpy.context.scene.ragdoll_tools_limits_mode
            mode=SetLimits.update_mode
            if mode==0:mode_name="Location and rotation"
            elif mode==1:mode_name="Location only"
            else:mode_name="Rotation only"

            lines=[]
            lines.append(f"Mode: {mode_name}")
            lines.append(" ")
            lines.append("Change mode: Shift-M")
            lines.append("Clear limits: C")
            lines.append(" ")
            lines.append("Toggle help: F1")
            lines.append("Exit: ESC")



            font_id=0
            font_size=15
            blf.size(font_id,font_size,72)
            blf.color(font_id,1,1,1,1)

            current_height=0
            margin=font_size*0.5
            start_x,start_y=0.1*bpy.context.region.width,0.1*bpy.context.region.height

            for line in reversed(lines):
                blf.position(font_id,start_x,start_y+current_height,0)
                blf.draw(font_id,line)

                w,h=blf.dimensions(font_id,line)
                current_height+=h+margin


    @staticmethod
    def limits_handler_source():

        if bpy.context.space_data.overlay.show_overlays:

            for holder in SetLimits.holders:holder.validate()
            valid_holders=[holder for holder in SetLimits.holders if holder.is_valid]

            for holder in valid_holders:
                rbc=holder.constraint.rigid_body_constraint
                visual_matrix=holder.node_a.matrix_world@holder.initial_matrix_a.inverted()@holder.constraint_initial_matrix

                segments=50
                radius=holder.constraint.empty_display_size

                rotation_axes=(
                mathutils.Vector((1.,0.,0.)),mathutils.Vector((0.,1.,0.)),mathutils.Vector((0.,0.,1.)))
                axes_to_rotate=(rotation_axes[1],rotation_axes[0],rotation_axes[1])
                colors=((1.0,0.0,0.0,0.3),(0.0,1.0,0.0,0.3),(0.0,0.0,1.0,0.3))
                solid_colors=((0.7,0.0,0.0,1.0),(0.0,0.7,0.0,1.0),(0.0,0.0,0.7,1.0))

                ang_limits=(
                    (('GENERIC','GENERIC_SPRING','PISTON'),rbc.use_limit_ang_x,rbc.limit_ang_x_lower,
                     rbc.limit_ang_x_upper),
                    (('GENERIC','GENERIC_SPRING'),rbc.use_limit_ang_y,rbc.limit_ang_y_lower,rbc.limit_ang_y_upper),
                    (('GENERIC','GENERIC_SPRING','HINGE'),rbc.use_limit_ang_z,rbc.limit_ang_z_lower,
                     rbc.limit_ang_z_upper),
                )

                linear_matrix=mathutils.Matrix.Translation(
                    visual_matrix.to_translation())@visual_matrix.to_quaternion().to_matrix().to_4x4()

                linear_limits=(
                    (('GENERIC','GENERIC_SPRING','PISTON','SLIDER'),rbc.use_limit_lin_x,
                     linear_matrix@mathutils.Vector((rbc.limit_lin_x_lower,0,0)),
                     linear_matrix@mathutils.Vector((rbc.limit_lin_x_upper,0,0))),
                    (('GENERIC','GENERIC_SPRING'),rbc.use_limit_lin_y,
                     linear_matrix@mathutils.Vector((0,rbc.limit_lin_y_lower,0)),
                     linear_matrix@mathutils.Vector((0,rbc.limit_lin_y_upper,0))),
                    (('GENERIC','GENERIC_SPRING'),rbc.use_limit_lin_z,
                     linear_matrix@mathutils.Vector((0,0,rbc.limit_lin_z_lower)),
                     linear_matrix@mathutils.Vector((0,0,rbc.limit_lin_z_upper)))
                )

                points=[]
                point_colors=[]

                for (allowed,use_angle,low_angle,up_angle),color,rotation_axis,axis_to_rotate in zip(ang_limits,colors,rotation_axes,axes_to_rotate):
                    if use_angle:

                        if rbc.type in allowed:

                            origin_point=tuple(visual_matrix@mathutils.Vector([0,0,0]))
                            origin_color=0.,0.,0.,0.

                            last_matrix=visual_matrix@mathutils.Matrix.Rotation(-low_angle,4,rotation_axis)
                            last_point=None

                            for x in range(segments+1):

                                point=tuple(last_matrix@(radius*axis_to_rotate))

                                if last_point is not None:
                                    point_colors.append(origin_color)
                                    points.append(origin_point)

                                    point_colors.append(color)
                                    points.append(point)

                                    point_colors.append(color)
                                    points.append(last_point)

                                last_matrix@=mathutils.Matrix.Rotation(-(up_angle-low_angle)/segments,4,rotation_axis)
                                last_point=point

                bgl.glEnable(bgl.GL_BLEND)
                bgl.glDisable(bgl.GL_DEPTH_TEST)

                render_batch=batch_for_shader(SetLimits.shader,'TRIS',{"pos":points,"color":point_colors})
                render_batch.draw(SetLimits.shader)

                points=[]
                point_colors=[]
                bgl.glDisable(bgl.GL_BLEND)
                bgl.glLineWidth(2)

                for (allowed,use_linear,low,up),color in zip(linear_limits,solid_colors):
                    if use_linear:
                        if rbc.type in allowed:
                            points.append(tuple(low))
                            point_colors.append(color)
                            points.append(tuple(up))
                            point_colors.append(color)

                render_batch=batch_for_shader(SetLimits.shader,'LINES',{"pos":points,"color":point_colors})
                render_batch.draw(SetLimits.shader)

                bgl.glPointSize(8)
                render_batch=batch_for_shader(SetLimits.shader,'POINTS',{"pos":points,"color":point_colors})
                render_batch.draw(SetLimits.shader)

    def clear_limits(self):
        update_pairs()


        for holder in SetLimits.holders:
            if holder.is_valid:
                hit_box=holder.node_b
                if bpy.context.mode=='POSE':
                    if hit_box in gv.hit_box_bone_dictionary:
                        if gv.hit_box_bone_dictionary[hit_box].bone.select:
                            holder.reset_constraint()
                if bpy.context.mode=='OBJECT':
                    if hit_box.select_get():
                        holder.reset_constraint()



    # def draw(self, context):
    #     layout = self.layout
    def stop_handler(self):
        # print(id(handler))
        keymap_items=bpy.context.window_manager.keyconfigs.active.keymaps['Screen Editing'].keymap_items
        if self.handler in bpy.app.handlers.depsgraph_update_post:
            bpy.app.handlers.depsgraph_update_post.remove(self.handler)

        if SetLimits.text_handler is not None:
            bpy.types.SpaceView3D.draw_handler_remove(SetLimits.text_handler,'WINDOW')
            SetLimits.text_handler=None

            bpy.types.SpaceView3D.draw_handler_remove(SetLimits.limits_handler,'WINDOW')
            SetLimits.limits_handler=None

        for item in SetLimits.keymap_items:
            keymap_items.remove(item)
        SetLimits.keymap_items=set()


    def execute(self, context):
        # print(id(handler))

        keymap_items=bpy.context.window_manager.keyconfigs.active.keymaps['Screen Editing'].keymap_items

        if self.mode=='START':
            # print("START")
            self.stop_handler()

            self.update_initial()
            bpy.app.handlers.depsgraph_update_post.append(self.handler)
            SetLimits.text_handler=bpy.types.SpaceView3D.draw_handler_add(self.text_handler_source,(None,None),'WINDOW','POST_PIXEL')
            SetLimits.limits_handler=bpy.types.SpaceView3D.draw_handler_add(SetLimits.limits_handler_source, (), 'WINDOW', 'POST_VIEW')

            item=keymap_items.new(SetLimits.bl_idname,'ESC','PRESS',head=True)
            item.properties.mode='STOP'
            SetLimits.keymap_items.add(item)

            item=keymap_items.new(SetLimits.bl_idname,'C','PRESS',head=True)
            item.properties.mode='CLEAR'
            SetLimits.keymap_items.add(item)


            item=keymap_items.new(SetLimits.bl_idname,'F1','PRESS',head=True)
            item.properties.mode='HELP'
            SetLimits.keymap_items.add(item)

            item=keymap_items.new(SetLimits.bl_idname,'M','PRESS',shift=True,head=True)
            item.properties.mode='MODE'
            SetLimits.keymap_items.add(item)


        if self.mode=='STOP':
            # print("STOP")
            self.stop_handler()


        if self.mode=='CLEAR':
            # print("CLEAR")
            self.clear_limits()

        if self.mode=='HELP':
            # print("HELP")
            SetLimits.show_help=not SetLimits.show_help

        if self.mode=='MODE':

            SetLimits.update_mode=(SetLimits.update_mode+1)%3

        bpy.ops.wm.redraw_timer(type='DRAW_WIN_SWAP',iterations=1)

        return {'FINISHED'}


def get_visible_bones(armature,selected_only=False):
    for pose_bone in armature.pose.bones:
        if not pose_bone.bone.hide:
            if pose_bone.bone.select or not selected_only:
                if any([bl and al for bl,al in zip(pose_bone.bone.layers,armature.data.layers)]):
                    yield pose_bone
def get_mesh_object():
    mesh_object=None
    for selected_object in bpy.context.selected_objects:
        if is_mesh(selected_object):
            if mesh_object is not None: return None
            else:mesh_object=selected_object


    return mesh_object
def get_armature():
    armature=None
    for selected_object in bpy.context.selected_objects:
        if is_armature(selected_object):
            if armature is not None:return None
            else: armature=selected_object
    return armature





class MakeVertexGroups(bpy.types.Operator):
    """Make vertex groups for selected bones if they don't exist"""
    bl_idname = "ragdoll.make_vertex_groups"
    bl_label = "Make vertex groups"
    bl_options = {'REGISTER', 'UNDO'}

    use_selected_bones_only: bpy.props.BoolProperty(name="Use selected bones only")

    @classmethod
    def poll(self, context):
        return context.mode in ('OBJECT','POSE')

    # def draw(self, context):
    #     layout = self.layout

    def execute(self, context):
        source_armature=get_armature()
        source_mesh=get_mesh_object()

        if None not in (source_armature,source_mesh):

            work_armature=bpy.data.objects.new("",bpy.data.armatures.new(""))

            bpy.context.collection.objects.link(work_armature)
            # work_armature.matrix_world=source_armature.matrix_world



            last_active=context.view_layer.objects.active
            last_mode=context.mode

            context.view_layer.objects.active=work_armature
            bpy.ops.object.mode_set(mode='EDIT')

            for pose_bone in get_visible_bones(source_armature,self.use_selected_bones_only):
                if pose_bone.name not in source_mesh.vertex_groups:
                    edit_bone=work_armature.data.edit_bones.new(pose_bone.name)
                    edit_bone.tail=pose_bone.bone.tail
                    edit_bone.head=pose_bone.bone.head
                    edit_bone.matrix=pose_bone.matrix_world
                    edit_bone.select=True

            work_armature.select_set(True)
            source_armature.select_set(False)

            bpy.ops.object.mode_set(mode='OBJECT')
            armature=source_mesh.modifiers.new("",'ARMATURE')
            armature.object=work_armature
            context.view_layer.update()


            context.view_layer.objects.active=source_mesh



            bpy.ops.object.mode_set(mode='WEIGHT_PAINT')
            bpy.ops.paint.weight_from_bones(type='AUTOMATIC')

            source_mesh.modifiers.remove(armature)

            to_remove=work_armature.data
            bpy.data.objects.remove(work_armature,do_unlink=True)
            bpy.data.armatures.remove(to_remove)

            source_armature.select_set(True)
            context.view_layer.objects.active=last_active
            bpy.ops.object.mode_set(mode=last_mode)

        return {'FINISHED'}


class ClearVertexGroups(bpy.types.Operator):
    """Clear vertex groups for selected bones only"""
    bl_idname = "ragdoll.clear_vertex_groups"
    bl_label = "Clear vertex groups"
    bl_options = {'REGISTER', 'UNDO'}

    use_selected_bones_only: bpy.props.BoolProperty(name="Use selected bones only")

    @classmethod
    def poll(self, context):
        return context.mode in ('OBJECT','POSE')

    # def draw(self, context):
    #     layout = self.layout

    def execute(self, context):
        source_armature=get_armature()
        source_mesh=get_mesh_object()

        if None not in (source_armature,source_mesh):


            for pose_bone in get_visible_bones(source_armature,self.use_selected_bones_only):
                if pose_bone.name in source_mesh.vertex_groups:
                    source_mesh.vertex_groups.remove(source_mesh.vertex_groups[pose_bone.name])


        return {'FINISHED'}


class FollowBone(bpy.types.Operator):
    """Make hitboxes follow bones"""
    bl_idname="ragdoll.follow_bone"
    bl_label="Follow bones"
    bl_options=set()

    def execute(self,context):
        bpy.ops.ragdoll.switch(mode='BONE')
        return {'FINISHED'}

class FollowHitbox(bpy.types.Operator):
    """Make bones follow hitboxes"""
    bl_idname="ragdoll.follow_hit_box"
    bl_label="Follow hitboxes"
    bl_options=set()

    def execute(self,context):
        bpy.ops.ragdoll.switch(mode='HIT_BOX')
        return {'FINISHED'}
class SwitchSource(bpy.types.Operator):
    """Switch followers"""
    bl_idname = "ragdoll.switch"
    bl_label = "Switch followers"
    bl_options = {'REGISTER', 'UNDO'}

    mode : bpy.props.EnumProperty(name="Settings",
                                          items=[
                                              ('HIT_BOX', "", "", 'MESH_DATA', 0),
                                              ('BONE',"","",'MESH_DATA',1),
                                          ]
                                          )

    # def draw(self, context):
    #     layout = self.layout


    def execute(self, context):
        update_pairs()
        # print(self.mode)
        if context.mode=='POSE':
            items=context.selected_pose_bones
        if context.mode=='OBJECT':
            items=context.selected_objects


        pairs=[]
        for item in items:
            pair=get_pair(item)
            if None not in pair:
                pairs.append(pair)


        if self.mode=='BONE':
            for hit_box,hit_bone in pairs:
                set_pair_head(hit_bone)


        if self.mode=='HIT_BOX':
            for hit_box,hit_bone in pairs:
                set_pair_head(hit_box)

        context.view_layer.update()
        return {'FINISHED'}



class Snap(bpy.types.Operator):
    """Snap constraints"""
    bl_idname = "ragdoll.snap"
    bl_label = "Snap constraints"
    bl_options = {'REGISTER', 'UNDO'}


    # def draw(self, context):
    #     layout = self.layout


    def execute(self, context):
        update_pairs()
        if context.mode=='POSE':
            items=context.selected_pose_bones
        if context.mode=='OBJECT':
            items=context.selected_objects


        hit_boxes=set()
        for item in items:
            hit_box,hit_bone=get_pair(item)
            if hit_box is not None:
                hit_boxes.add(hit_box)

        for constraint in bpy.data.objects:
            rbc=constraint.rigid_body_constraint
            if rbc is not None:
                if hit_boxes.issuperset((rbc.object1,rbc.object2)) or  constraint.select_get() and None not in get_pair(rbc.object1) :

                    constraint.matrix_world=rbc.object2.matrix_world@constraint.ragdoll_tools_other_rest_matrix.inverted()@constraint.ragdoll_tools_own_rest_matrix



        # bpy.ops.wm.redraw_timer(type='DRAW_WIN_SWAP', iterations=1)

        return {'FINISHED'}














class UnlinkHitBoxAndBone(bpy.types.Operator):
    """Disconnect bones from hitboxes"""
    bl_idname = "ragdoll.unlink_hit_boxes_and_bones"
    bl_label = "Disconnect"
    bl_options = {'REGISTER', 'UNDO'}
    # bl_options={'REGISTER','UNDO_GROUPED','UNDO','BLOCKING'}

    use_selected_bones_only : bpy.props.BoolProperty(name="Use selected bones only")

    delete_hit_boxes : bpy.props.BoolProperty(name="Delete hitboxes", description="Delete hitboxes", default=False)
    delete_constraints : bpy.props.BoolProperty(name="Delete related constraints", description="Delete related constraints", default=True)

    def draw(self, context):
        layout = self.layout
        layout.prop(self, "use_selected_bones_only")
        layout.prop(self, "delete_hit_boxes")
        layout.prop(self, "delete_constraints")

    def check(self,context):
        # print("CHECK")
        # time.sleep(1)
        return True

    # def invoke(self,context,event):
    #     # bpy.context.scene.rigidbody_world.enabled=False
    #     # bpy.ops.ragdoll.clear_cache()
    #     return {'FINISHED'}

    @classmethod
    def poll(self, context):
        return context.mode in ('OBJECT','POSE')


    def execute(self, context):
        # time.sleep(1)
        # context.view_layer.update()
        update_pairs()


        hit_bone_set = set()
        hit_box_set = set()

        unlinked_items = set()

        mesh_set=set()

        def add_to_set(item):
            hit_box, hit_bone = get_pair(item)
            if None not in (hit_box, hit_bone):
                hit_box_set.add(hit_box)
                hit_bone_set.add(hit_bone)
            else:
                unlinked_items.add(item)
            if is_mesh(item):
                mesh_set.add(item)

        if context.mode == 'POSE':
            if self.use_selected_bones_only:
                selected_pose_bones = context.selected_pose_bones
            else:
                selected_pose_bones = context.visible_pose_bones
            for pose_bone in selected_pose_bones:
                add_to_set(pose_bone)


        else:
            for selected_object in context.selected_objects:
                if context.mode != 'POSE':
                    if is_armature(selected_object):
                        for pose_bone in selected_object.pose.bones:
                            if not pose_bone.bone.hide:
                                if pose_bone.bone.select or not self.use_selected_bones_only:
                                    if any([bl and al for bl, al in zip(pose_bone.bone.layers, selected_object.data.layers)]):
                                        add_to_set(pose_bone)

                if is_mesh(selected_object):
                    add_to_set(selected_object)

        before_matrices=[item.matrix_world.copy() for item in itertools.chain(hit_box_set,hit_bone_set)]

        remove_rig(hit_box_set)
        remove_rig(hit_bone_set)
        remove_rig(unlinked_items)

        for item,before_matrix in zip(itertools.chain(hit_box_set,hit_bone_set),before_matrices):item.matrix_world=before_matrix

        # context.view_layer.update()
        # bpy.ops.ragdoll.clear_cache()
        # context.view_layer.update()

        # print(id(bpy.context.scene.rigidbody_world.constraints.objects) , id(context.scene.rigidbody_world.constraints.objects))

        to_remove=[]

        if self.delete_constraints:



            constraints_to_delete=set()
            if context.scene.rigidbody_world!=None:
                if context.scene.rigidbody_world.constraints!=None:
                    for item in context.scene.rigidbody_world.constraints.objects:
                        if item.rigid_body_constraint!=None:
                            if item.rigid_body_constraint.object1 in hit_box_set or item.rigid_body_constraint.object2 in hit_box_set:
                                # print("AAA",item)
                                constraints_to_delete.add(item)




            to_remove.extend(constraints_to_delete)



        if self.delete_hit_boxes:
            to_remove.extend(hit_box_set)


        override=bpy.context.copy()
        override["selected_objects"]=to_remove


        last_mode=bpy.context.mode
        bpy.ops.object.mode_set(mode='OBJECT')
        bpy.ops.object.delete(override,'EXEC_DEFAULT',False)
        if last_mode!='OBJECT':
            bpy.ops.object.mode_set(mode=last_mode)


        # bpy.ops.wm.redraw_timer(type='DRAW_WIN_SWAP', iterations=1)

        return {'FINISHED'}


@inner_pgroups_to_pointers
class SetRBCSettings(bpy.types.Operator):
    """Copy rigid body constraint settings from active to selected"""
    bl_idname = "ragdoll.set_rigid_body_constraint_settings"
    bl_label = "Copy rigid body constraint settings"
    bl_options = {'REGISTER', 'UNDO'}

    @inner_pgroups_to_pointers
    class object(bpy.types.PropertyGroup):
        @pgroup_from_type(bpy.types.RigidBodyConstraint, **default_rbc_settings)
        class rigid_body_constraint(bpy.types.PropertyGroup): pass



    def check(self, context):
        # time.sleep(0.05)
        return True

    def draw(self, context):
        layout = self.layout
        rbc_panel(layout,self)

    active_constraint_name=None
    selected_constraint_names=[]

    def invoke(self, context, event):
        # print("INVOKE",id(context.object))

        update_pairs()

        SetRBCSettings.active_constraint_name=None
        SetRBCSettings.selected_constraint_names=[]

        active_constraint=None
        selected_constraints=[]

        if context.mode=='POSE':
            object_constraint_dictionary={constraint.rigid_body_constraint.object2:constraint for constraint in
                                          bpy.data.objects if constraint.rigid_body_constraint is not None}

            if context.active_pose_bone in gv.linked_items:
                active_object=gv.bone_hit_box_dictionary[context.active_pose_bone]
                if active_object in object_constraint_dictionary:
                    active_constraint=object_constraint_dictionary[active_object]

            selected_objects=[gv.bone_hit_box_dictionary[bone] for bone in context.selected_pose_bones if bone in gv.linked_items]
            selected_constraints=[object_constraint_dictionary[selected_object] for selected_object in selected_objects if selected_object in object_constraint_dictionary]

        if context.mode=='OBJECT':
            if context.object.rigid_body_constraint is not None:
                active_constraint=context.object
            selected_constraints=[selected_object for selected_object in context.selected_objects if selected_object.rigid_body_constraint is not None]



        if active_constraint is not None:
            copy_settings(active_constraint.rigid_body_constraint, self.object.rigid_body_constraint,self.object.rigid_body_constraint.property_keys)

            SetRBCSettings.active_constraint_name=active_constraint.name
            SetRBCSettings.selected_constraint_names=[selected_constraint.name for selected_constraint in selected_constraints]



        return self.execute(context)

    def execute(self, context):


        # print("EXECUTE",id(context.object))

        if SetRBCSettings.active_constraint_name is not None:

            selected_constraints=[bpy.data.objects[selected_constraint] for selected_constraint in SetRBCSettings.selected_constraint_names]

            for selected_object in selected_constraints:

                copy_settings(self.object.rigid_body_constraint, selected_object.rigid_body_constraint,self.object.rigid_body_constraint.property_keys)


        return {'FINISHED'}


@inner_pgroups_to_pointers
class SetRBSettings(bpy.types.Operator):
    """Copy rigid body settings from active to selected"""
    bl_idname = "ragdoll.set_rigid_body_settings"
    bl_label = "Copy rigid body settings"
    bl_options = {'REGISTER', 'UNDO'}

    @inner_pgroups_to_pointers
    class object(bpy.types.PropertyGroup):
        @pgroup_from_type(bpy.types.RigidBodyObject)
        class rigid_body(bpy.types.PropertyGroup): pass

    active_object_name=None
    selected_object_names=[]

    def check(self, context):
        return True

    def draw(self, context):
        rb_panel(self.layout,self)


    def invoke(self, context, event):
        update_pairs()

        SetRBSettings.active_object_name=None
        SetRBSettings.selected_object_names=[]

        active_object=None
        selected_objects=[]


        if context.mode=='POSE':

            if context.active_pose_bone in gv.linked_items:
                active_object=gv.bone_hit_box_dictionary[context.active_pose_bone]
                if active_object.rigid_body is None:
                    active_object=None
            if active_object is not None:
                selected_objects=[gv.bone_hit_box_dictionary[bone] for bone in context.selected_pose_bones if bone in gv.linked_items]
                selected_objects=[selected_object for selected_object in selected_objects if selected_object.rigid_body is not None]

        if context.mode=='OBJECT':
            if context.object is not None:
                if context.object.rigid_body is not None:
                    active_object=context.object
                    selected_objects=[selected_object for selected_object in context.selected_objects if selected_object.rigid_body is not None]

        if active_object is not None:
            copy_settings(active_object.rigid_body,self.object.rigid_body,self.object.rigid_body.property_keys)

            SetRBSettings.active_object_name=active_object.name
            SetRBSettings.selected_object_names=[selected_object.name for selected_object in selected_objects]


        return self.execute(context)

    def execute(self, context):

        if SetRBSettings.active_object_name is not None:
            selected_objects=[bpy.data.objects[name] for name in SetRBSettings.selected_object_names]
            for selected_object in selected_objects:
                copy_settings(self.object.rigid_body, selected_object.rigid_body, self.object.rigid_body.property_keys)

        return {'FINISHED'}







@inner_pgroups_to_pointers
class MakeRigidBodyConstraints(bpy.types.Operator):
    """Connect selected rigid bodies with rigid body constraints"""
    bl_idname = "ragdoll.make_rigid_body_constraints"
    bl_label = "Generate rigid body constraints"
    bl_options = {'REGISTER', 'UNDO'}

    show_in_front: bpy.props.BoolProperty(name="Make the object draw in front of others")

    constraint_collection: bpy.props.StringProperty(name="Constraint collection",default=constraint_group_name,
                                                    description="Collection name of generated constraints")

    settings_tab : bpy.props.EnumProperty(name="Settings",
                                          items=[
                                              ('GENERAL', "Generation settings", "Settings of generated constraints ", 'MESH_DATA', 0),
                                              ('RIGID_BODY', "Physics settings", "Rigid body settings of generated constraints",'MESH_ICOSPHERE', 1)
                                          ]
                                          )


    @inner_pgroups_to_pointers
    class object(bpy.types.PropertyGroup):
        @pgroup_from_type(bpy.types.RigidBodyConstraint, **default_rbc_settings)
        class rigid_body_constraint(bpy.types.PropertyGroup): pass




    class constraint_settings(bpy.types.PropertyGroup):


        connection_pattern : bpy.props.EnumProperty(name="Connection pattern", default='SELECTED_TO_ACTIVE',
                                                    items=[
                                                        ('SELECTED_TO_ACTIVE', "Selected to active",
                                                         "Connect selected objects to active with rigid body constraint",
                                                         'NONE', 0),
                                                        ('CHAIN_BY_DISTANCE', "Chain by distance",
                                                         "Connect objects as a chain based on distance,starting at the active object",
                                                         'NONE', 1),
                                                        ('ALL', "All to all",
                                                         "Connect every object with all the others",
                                                         'NONE', 2),
                                                    ]
                                                    )

        connection_pattern_with_bones : bpy.props.EnumProperty(name="Connection pattern", default='FROM_BONES',
                                                               items=[
                                                                   ('SELECTED_TO_ACTIVE', "Selected to active",
                                                                    "Connect selected objects to active with rigid body constraint",
                                                                    'NONE', 0),
                                                                   ('CHAIN_BY_DISTANCE', "Chain by distance",
                                                                    "Connect objects as a chain based on distance,starting at the active object",
                                                                    'NONE', 1),

                                                                   ('ALL', "All to all",
                                                                    "Connect every object with all others",
                                                                    'NONE', 2),

                                                                   ('FROM_BONES', "Using bones hierarchy",
                                                                    "Use hierarchy of connected bones to connect objects with rigid body constraints",
                                                                    'NONE', 3),
                                                               ]
                                                               )


        #
        # constraint_location : bpy.props.EnumProperty(name="Location", default='CHILD',
        #                                              items=[
        #                                                  ('CHILD', "Child", "Child", 'NONE', 0),
        #                                                  ('PARENT', "Parent", "Parent", 'NONE', 1),
        #                                                  ('CENTER', "Between", "Between", 'NONE', 2),
        #                                              ]
        #                                              )



        constraint_location_hierarchy: bpy.props.EnumProperty(name="Location",default='CHILD',
                                                    items=[
                                                        ('CHILD',"Child","Child",'NONE',0),
                                                        ('PARENT',"Parent","Parent",'NONE',1),
                                                        ('CENTER',"Between","Between",'NONE',2),
                                                    ]
                                                    )



        constraint_location_selected: bpy.props.EnumProperty(name="Location",default='CHILD',
                                                    items=[
                                                        ('CHILD',"Selected","Selected",'NONE',0),
                                                        ('PARENT',"Active","Active",'NONE',1),
                                                        ('CENTER',"Between","Between",'NONE',2),
                                                    ]
                                                    )

        constraint_location_other: bpy.props.EnumProperty(name="Location",default='CHILD',
                                                    items=[
                                                        ('CHILD',"Second","Second",'NONE',0),
                                                        ('PARENT',"First","First",'NONE',1),
                                                        ('CENTER',"Between","Between",'NONE',2),
                                                    ]
                                                    )




        origins_items = [
            ('ORIGIN', "Hitbox", "Object origin", 'NONE', 0),
            ('HEAD', "Bone head", "Head of connected bone", 'NONE', 1),
            ('TAIL', "Bone tail", "Tail of connected bone", 'NONE', 2),
        ]

        origin_a : bpy.props.EnumProperty(name="Origin", items=origins_items, default='HEAD')
        origin_b : bpy.props.EnumProperty(name="Origin", items=origins_items, default='HEAD')

        # constraint_orientation : bpy.props.EnumProperty(name="Orientation", default='CHILD',
        #                                                 items=[
        #                                                     ('CHILD', "Child", "Child", 'NONE', 0),
        #                                                     ('PARENT', "Parent", "Parent", 'NONE', 1),
        #                                                 ]
        #                                                 )


        constraint_orientation_hierarchy : bpy.props.EnumProperty(name="Orientation", default='CHILD',
                                                        items=[
                                                            ('CHILD', "Child", "Child", 'NONE', 0),
                                                            ('PARENT', "Parent", "Parent", 'NONE', 1),
                                                        ]
                                                        )



        constraint_orientation_selected : bpy.props.EnumProperty(name="Orientation", default='CHILD',
                                                        items=[
                                                            ('CHILD', "Selected", "Selected", 'NONE', 0),
                                                            ('PARENT', "Active", "Active", 'NONE', 1),
                                                        ]
                                                        )


        constraint_orientation_other : bpy.props.EnumProperty(name="Orientation", default='CHILD',
                                                        items=[
                                                            ('CHILD', "Second", "Second", 'NONE', 0),
                                                            ('PARENT', "First", "First", 'NONE', 1),
                                                        ]
                                                        )



        # constraint_orientation_with_bones : bpy.props.EnumProperty(name="Orientation", default='CHILD_BONE',
        #                                                            items=[
        #                                                                ('CHILD', "Child", "Child", 'NONE', 0),
        #                                                                ('PARENT', "Parent", "Parent", 'NONE', 1),
        #                                                                ('CHILD_BONE', "Child bone",
        #                                                                 "Child bone", 'NONE', 2),
        #                                                                ('PARENT_BONE', "Parent bone",
        #                                                                 "Parent bone", 'NONE', 3),
        #                                                            ]
        #                                                            )

        constraint_orientation_with_bones_hierarchy : bpy.props.EnumProperty(name="Orientation", default='CHILD_BONE',
                                                                   items=[
                                                                       ('CHILD', "Child hitbox", "Child hitbox", 'NONE', 0),
                                                                       ('PARENT', "Parent hitbox", "Parent hitbox", 'NONE', 1),
                                                                       ('CHILD_BONE', "Child bone",
                                                                        "Child bone", 'NONE', 2),
                                                                       ('PARENT_BONE', "Parent bone",
                                                                        "Parent bone", 'NONE', 3),
                                                                   ]
                                                                   )

        constraint_orientation_with_bones_selected: bpy.props.EnumProperty(name="Orientation",default='CHILD_BONE',
                                                                            items=[
                                                                                ('CHILD',"Selected hitbox","Selected hitbox",'NONE',0),
                                                                                ('PARENT',"Active hitbox","Active hitbox",'NONE',1),
                                                                                ('CHILD_BONE',"Selected bone",
                                                                                 "Selected bone",'NONE',2),
                                                                                ('PARENT_BONE',"Active bone",
                                                                                 "Active bone",'NONE',3),
                                                                            ]
                                                                            )


        constraint_orientation_with_bones_other: bpy.props.EnumProperty(name="Orientation",default='CHILD_BONE',
                                                                            items=[
                                                                                ('CHILD',"Second hitbox","Second hitbox",'NONE',0),
                                                                                ('PARENT',"First hitbox","First hitbox",'NONE',1),
                                                                                ('CHILD_BONE',"Second bone",
                                                                                 "Second bone",'NONE',2),
                                                                                ('PARENT_BONE',"First bone",
                                                                                 "First bone",'NONE',3),
                                                                            ]
                                                                            )



        empty_display_type : prop_from_type(bpy.types.Object.bl_rna.properties["empty_display_type"],name="Display",
                                            default='SPHERE')
        empty_display_size : prop_from_type(bpy.types.Object.bl_rna.properties["empty_display_size"],default=0.2)

    @classmethod
    def poll(self, context):
        # if context.mode not in ('OBJECT','POSE'):return False

        update_pairs()

        self.has_connected_bones = False
        self.linked_meshes = {}
        self.linked_bones = {}
        self.meshes = []

        if context.mode == 'POSE':
            for pose_bone in context.selected_pose_bones:
                hit_box, hit_bone = get_pair(pose_bone)
                if None not in (hit_box, hit_bone):
                    if hit_box.rigid_body != None:
                        self.linked_meshes[hit_box] = hit_bone
                        self.linked_bones[hit_bone] = hit_box
                        self.meshes.append(hit_box)
        else:
            for selected_object in context.selected_objects:
                # print(selected_object.rigidbody)
                if is_mesh(selected_object) :
                    if selected_object.rigid_body != None:
                        self.meshes.append(selected_object)
                hit_box, hit_bone = get_pair(selected_object)
                if None not in (hit_box, hit_bone):
                    self.linked_meshes[hit_box] = hit_bone
                    self.linked_bones[hit_bone] = hit_box

        if len(self.meshes) >= 2:
            if len(self.linked_meshes) != 0:
                self.has_connected_bones = True
            return True
        else:
            return False

    def get_transform(self, parent_object, child_object):

        # constraint_location
        # constraint_orientation
        # constraint_orientation_with_bones



        if self.has_connected_bones:
            work_connection_pattern=self.constraint_settings.connection_pattern_with_bones
        else:
            work_connection_pattern=self.constraint_settings.connection_pattern


        if work_connection_pattern=='FROM_BONES':
            work_constraint_location=self.constraint_settings.constraint_location_hierarchy
            work_constraint_orientation=self.constraint_settings.constraint_orientation_hierarchy
            work_constraint_orientation_with_bones=self.constraint_settings.constraint_orientation_with_bones_hierarchy
        elif work_connection_pattern=='SELECTED_TO_ACTIVE':
            work_constraint_location=self.constraint_settings.constraint_location_selected
            work_constraint_orientation=self.constraint_settings.constraint_orientation_selected
            work_constraint_orientation_with_bones=self.constraint_settings.constraint_orientation_with_bones_selected
        else:
            work_constraint_location=self.constraint_settings.constraint_location_other
            work_constraint_orientation=self.constraint_settings.constraint_orientation_other
            work_constraint_orientation_with_bones=self.constraint_settings.constraint_orientation_with_bones_other


        location = None
        rotation = None

        if self.has_connected_bones:
            if work_constraint_orientation_with_bones == 'PARENT_BONE' and parent_object in self.linked_meshes:
                rotation = self.linked_meshes[parent_object].matrix_world.to_quaternion()
            if work_constraint_orientation_with_bones == 'CHILD_BONE' and child_object in self.linked_meshes:
                rotation = self.linked_meshes[child_object].matrix_world.to_quaternion()

            if work_constraint_orientation_with_bones == 'PARENT':
                rotation = parent_object.matrix_world.to_quaternion()
            if work_constraint_orientation_with_bones == 'CHILD':
                rotation = child_object.matrix_world.to_quaternion()
            pass
        else:
            if work_constraint_orientation == 'PARENT':
                rotation = parent_object.matrix_world.to_quaternion()
            if work_constraint_orientation == 'CHILD':
                rotation = child_object.matrix_world.to_quaternion()

        locations = [None, None]
        hit_boxes = (parent_object, child_object)
        if work_constraint_location == 'CENTER':
            origins = (self.constraint_settings.origin_a, self.constraint_settings.origin_b)
        else:
            origins = (self.constraint_settings.origin_a, self.constraint_settings.origin_a)

        for x, (origin, hit_box) in enumerate(zip(origins, hit_boxes)):

            if origin == 'ORIGIN' or hit_box not in self.linked_meshes:
                locations[x] = hit_box.matrix_world.to_translation()

            else:
                if origin == 'HEAD':
                    locations[x] = self.linked_meshes[hit_box].head_world
                if origin == 'TAIL':
                    locations[x] = self.linked_meshes[hit_box].tail_world

        if work_constraint_location == 'PARENT':
            location = locations[0]

        if work_constraint_location == 'CHILD':
            location = locations[1]

        if work_constraint_location == 'CENTER':
            location = (locations[0] + locations[1]) * 0.5

        out_matrix = mathutils.Matrix.Translation(location) @ mathutils.Matrix.Rotation(rotation.angle, 4,rotation.axis)

        return out_matrix

    def make_const(self, parent_object, child_object, context):

        const_empty=self.init_const.copy()
        const_empty.name="const_empty"

        const_empty.show_in_front=self.show_in_front

        # context.scene.objects.link(const_empty)
        self.collection.objects.link(const_empty)
        context.scene.rigidbody_world.constraints.objects.update()
        if const_empty not in context.scene.rigidbody_world.constraints.objects.values():
            context.scene.rigidbody_world.constraints.objects.link(const_empty)

        const_empty.rigid_body_constraint.object1 = parent_object
        const_empty.rigid_body_constraint.object2 = child_object

        const_empty.matrix_world = self.get_transform(parent_object, child_object)


        const_empty.ragdoll_tools_own_rest_matrix=flatten(const_empty.matrix_world)
        const_empty.ragdoll_tools_other_rest_matrix=flatten(child_object.matrix_world)

        const_empty.empty_display_size*=sum(child_object.dimensions)/3



    def execute(self, context):
        context.view_layer.update()
        update_pairs()



        if context.scene.rigidbody_world is None:
            bpy.ops.rigidbody.world_add()
        last_active = context.object



        self.init_const = bpy.data.objects.new("", None)
        self.init_const.empty_display_type = self.constraint_settings.empty_display_type
        self.init_const.empty_display_size = self.constraint_settings.empty_display_size



        if self.constraint_collection in bpy.data.collections:
            collection=bpy.data.collections[self.constraint_collection]
        else:
            collection=bpy.data.collections.new(self.constraint_collection)
            context.scene.collection.children.link(collection)

        collection.objects.link(self.init_const)

        self.collection=collection


        context.view_layer.objects.active=self.init_const
        context.view_layer.update()

        bpy.ops.rigidbody.constraint_add()
        context.view_layer.objects.active = last_active



        copy_settings(self.object.rigid_body_constraint, self.init_const.rigid_body_constraint,self.object.rigid_body_constraint.property_keys)




        if self.has_connected_bones:
            pattern = self.constraint_settings.connection_pattern_with_bones
        else:
            pattern = self.constraint_settings.connection_pattern

        if pattern == 'SELECTED_TO_ACTIVE':


            if bpy.context.mode=='POSE':
                # print(context.active_pose_bone)
                if context.active_pose_bone in self.linked_bones:
                    parent_object=self.linked_bones[context.active_pose_bone]
                else:parent_object=None
            else:
                parent_object = context.object


            if is_mesh(parent_object) and parent_object.rigid_body != None:
                for child_object in self.meshes:
                    if child_object != parent_object:
                        self.make_const(parent_object, child_object, context)

        if pattern == 'CHAIN_BY_DISTANCE':
            if is_mesh(last_active) and context.object.rigid_body != None:
                parent_object = last_active
            else:
                parent_object = self.meshes[0]

            tmp_hit_boxes = set(self.meshes)
            if parent_object in tmp_hit_boxes:
                tmp_hit_boxes.remove(parent_object)
            while len(tmp_hit_boxes) != 0:
                child_object = min(tmp_hit_boxes, key=lambda v: length(v.location_world, parent_object.location_world))
                self.make_const(parent_object, child_object, context)
                tmp_hit_boxes.remove(child_object)
                parent_object = child_object

        if pattern == 'FROM_BONES':
            for pose_bone in self.linked_bones:
                if pose_bone.parent in self.linked_bones:
                    parent_object = self.linked_bones[pose_bone.parent]
                    child_object = self.linked_bones[pose_bone]
                    self.make_const(parent_object, child_object, context)


        if pattern == 'ALL':
            for x in range(len(self.meshes)):
                for y in range(x+1,len(self.meshes)):
                    self.make_const(self.meshes[x], self.meshes[y], context)

        bpy.data.objects.remove(self.init_const, do_unlink=True)



        return {'FINISHED'}

    # def invoke(self,context,event):
    #     print("INVOKE")
    #     return {'FINISHED'}

    def check(self, context):
        return True

    def draw(self, context):

        # constraint_location
        # constraint_orientation
        # constraint_orientation_with_bones
        #


        if self.has_connected_bones:
            work_connection_pattern=self.constraint_settings.connection_pattern_with_bones
            connection_pattern_name="connection_pattern_with_bones"
        else:
            work_connection_pattern=self.constraint_settings.connection_pattern
            connection_pattern_name="connection_pattern"


        if work_connection_pattern=='FROM_BONES':

            parent_label_name="Parent"
            child_label_name="Child"
            constraint_location_name="constraint_location_hierarchy"
            constraint_orientation_name="constraint_orientation_hierarchy"
            constraint_orientation_with_bones_name="constraint_orientation_with_bones_hierarchy"
            work_constraint_location=self.constraint_settings.constraint_location_hierarchy
        elif work_connection_pattern=='SELECTED_TO_ACTIVE':
            parent_label_name="Active"
            child_label_name="Selected"
            constraint_location_name="constraint_location_selected"
            constraint_orientation_name="constraint_orientation_selected"
            constraint_orientation_with_bones_name="constraint_orientation_with_bones_selected"
            work_constraint_location=self.constraint_settings.constraint_location_selected
        else:
            parent_label_name="First"
            child_label_name="Second"
            constraint_location_name="constraint_location_other"
            constraint_orientation_name="constraint_orientation_other"
            constraint_orientation_with_bones_name="constraint_orientation_with_bones_other"
            work_constraint_location=self.constraint_settings.constraint_location_other



        layout = self.layout
        layout.prop(self,"settings_tab",text="")
        layout.separator()



        if self.settings_tab=='GENERAL':


            layout.prop(self.constraint_settings,connection_pattern_name,text="Connect")

            layout.separator()

            col=layout.column(align=True)

            col.label(text="Location:")

            if self.has_connected_bones:
                if work_constraint_location=='CENTER':
                    col.prop(self.constraint_settings,constraint_location_name,text="")

                    row=col.row()

                    col_a=row.column()
                    col_a.label(text=parent_label_name)
                    col_a.prop(self.constraint_settings,"origin_a",text="")

                    col_b=row.column()
                    col_b.label(text=child_label_name)
                    col_b.prop(self.constraint_settings,"origin_b",text="")
                else:
                    col.prop(self.constraint_settings,constraint_location_name,text="")
                    col.prop(self.constraint_settings,"origin_a",text="")
            else:
                col.prop(self.constraint_settings,constraint_location_name,text="")


            layout.separator()

            layout.label(text="Orientation:")
            if self.has_connected_bones:
                layout.prop(self.constraint_settings,constraint_orientation_with_bones_name,text="")
            else:
                layout.prop(self.constraint_settings,constraint_orientation_name,text="")

            layout.separator()
            layout.separator()

            layout.prop(self,"constraint_collection",text="Collection")

            layout.prop(self.constraint_settings,"empty_display_type",text="Display As")
            layout.prop(self.constraint_settings,"empty_display_size",text="Size")
            layout.prop(self,"show_in_front",text="In Front")



        else:

            rbc_panel(layout,self)


@inner_pgroups_to_pointers
class GenerateHitBoxes(bpy.types.Operator):
    """Make hitboxes"""
    bl_idname = "ragdoll.generate_hit_boxes"
    bl_label = "Make hitboxes"
    bl_options = {'REGISTER', 'UNDO'}

    def get_hit_box_name(self,pose_bone):
        return pose_bone.id_data.name + "_" + pose_bone.name + "_hitbox"

    def make_hit_box(self, mesh, pose_bone, context):
        out_object = bpy.data.objects.new(self.get_hit_box_name(pose_bone), mesh)
        origin_shift = mathutils.Vector([0, (pose_bone.bone.center - pose_bone.bone.head).length, 0])
        out_object.matrix_world = pose_bone.matrix_world @ mathutils.Matrix.Translation(origin_shift)
        out_object.data.transform(mathutils.Matrix.Translation(-origin_shift))
        # context.scene.objects.link(out_object)
        # context.scene.collection.objects.link(out_object)
        return out_object

    def capsule_mesh(self, bone,matrix=None):
        if self.mesh_settings.use_envelope:
            if bone.parent != None and bone.use_connect:
                head_radius = bone.parent.tail_radius * self.mesh_settings.head_radius_mlt + bone.envelope_distance * self.mesh_settings.envelope_distance_mlt
            else:
                head_radius = bone.head_radius * self.mesh_settings.head_radius_mlt + bone.envelope_distance * self.mesh_settings.envelope_distance_mlt
            tail_radius = bone.tail_radius * self.mesh_settings.tail_radius_mlt + bone.envelope_distance * self.mesh_settings.envelope_distance_mlt
        else:
            head_radius = self.mesh_settings.head_radius
            tail_radius = self.mesh_settings.tail_radius
        head_shift = bone.length * self.mesh_settings.head_shift
        tail_shift = bone.length * self.mesh_settings.tail_shift

        u_segments = self.mesh_settings.segments
        v_segments = self.mesh_settings.segments

        out_mesh = None
        if head_radius != 0 and tail_radius != 0 and u_segments >= 2 and v_segments >= 2:
            out_name = bone.id_data.name + "_" + bone.name + "_hitbox"
            out_mesh = bpy.data.meshes.new(out_name)

            eps = 0.0001

            bm = bmesh.new()
            bm.from_mesh(out_mesh)

            head_sphere = bmesh.ops.create_uvsphere(bm, u_segments=u_segments, v_segments=v_segments,
                                                    diameter=head_radius)
            to_delete = [v for v in head_sphere["verts"] if v.co.z > eps]
            bmesh.ops.translate(bm, verts=head_sphere["verts"], vec=mathutils.Vector([0, 0, head_shift]))
            bmesh.ops.delete(bm, geom=to_delete, context='VERTS')

            tail_sphere = bmesh.ops.create_uvsphere(bm, u_segments=u_segments, v_segments=v_segments,diameter=tail_radius)
            to_delete = [v for v in tail_sphere["verts"] if v.co.z < -eps]
            bmesh.ops.translate(bm, verts=tail_sphere["verts"],vec=mathutils.Vector([0, 0, bone.vector.length - tail_shift]))
            bmesh.ops.delete(bm, geom=to_delete, context='VERTS')

            boundary = [edge for edge in bm.edges if edge.is_boundary]
            bmesh.ops.bridge_loops(bm, edges=boundary)

            bmesh.ops.transform(bm, verts=bm.verts, matrix=mathutils.Matrix.Rotation(-math.pi / 2, 4, 'X'))

            if matrix!=None:
                bmesh.ops.transform(bm, verts=bm.verts, matrix=matrix)

            bm.to_mesh(out_mesh)

            bm.free()

        return out_mesh


    def bounding_box_mesh(self,source):
        out_name=source.name+"_bounding_box"
        out_mesh=bpy.data.meshes.new(out_name)

        dimensions=source.dimensions

        if 0 in dimensions: return None


        bm=bmesh.new()
        bm.from_mesh(out_mesh)
        matrix=mathutils.Matrix.Identity(4)
        # matrix=source.matrix_basis.copy()
        # matrix.col[3].xyz=source.matrix_world.col[3].xyz

        matrix.col[0].xyz*=dimensions[0]
        matrix.col[1].xyz*=dimensions[1]
        matrix.col[2].xyz*=dimensions[2]

        bmesh.ops.create_cube(bm,size=1,matrix=matrix)


        min_dimension=min(dimensions)
        split_counts=[min(50,round(dimensions[x]/min_dimension)) for x in range(3)]


        for axis in range(3):
            start=-dimensions[axis]/2
            step=dimensions[axis]/split_counts[axis]
            for x in range(1,split_counts[axis]):
                normal=mathutils.Vector()
                normal[axis]=1
                point=mathutils.Vector()
                point[axis]=start+x*step

                geom=list(bm.edges)+list(bm.faces)
                bmesh.ops.bisect_plane(bm,geom=geom,dist=0.0,plane_co=point,plane_no=normal,use_snap_center=False,clear_outer=False,clear_inner=False)


        bound_center=sum((mathutils.Vector(point) for point in source.bound_box),mathutils.Vector())/8

        bmesh.ops.translate(bm,vec=bound_center,space=mathutils.Matrix.Identity(4),verts=bm.verts)

        for f in bm.faces:
            f.smooth=True



        bm.to_mesh(out_mesh)
        bm.free()

        return out_mesh



    def box_mesh(self, bone,matrix=None):
        if self.mesh_settings.use_envelope:
            if bone.parent != None and bone.use_connect:
                head_radius = bone.parent.tail_radius * self.mesh_settings.head_radius_mlt + bone.envelope_distance * self.mesh_settings.envelope_distance_mlt
            else:
                head_radius = bone.head_radius * self.mesh_settings.head_radius_mlt + bone.envelope_distance * self.mesh_settings.envelope_distance_mlt
            tail_radius = bone.tail_radius * self.mesh_settings.tail_radius_mlt + bone.envelope_distance * self.mesh_settings.envelope_distance_mlt
        else:
            head_radius = self.mesh_settings.head_radius
            tail_radius = self.mesh_settings.tail_radius
        head_shift = bone.length * self.mesh_settings.head_shift
        tail_shift = bone.length * self.mesh_settings.tail_shift

        u_segments = self.mesh_settings.segments
        v_segments = self.mesh_settings.segments

        out_mesh = None
        if head_radius != 0 and tail_radius != 0 and u_segments >= 2 and v_segments >= 2:
            out_name = bone.id_data.name + "_" + bone.name + "_hitbox"
            out_mesh = bpy.data.meshes.new(out_name)

            eps = 0.0001

            bm = bmesh.new()
            bm.from_mesh(out_mesh)

            head_sphere = bmesh.ops.create_grid(bm, size=head_radius)
            bmesh.ops.reverse_faces(bm, faces=bm.faces)
            to_delete = [v for v in head_sphere["verts"] if v.co.z > eps]
            bmesh.ops.translate(bm, verts=head_sphere["verts"], vec=mathutils.Vector([0, 0, head_shift]))
            bmesh.ops.delete(bm, geom=to_delete, context='VERTS')

            tail_sphere = bmesh.ops.create_grid(bm, size=tail_radius)
            to_delete = [v for v in tail_sphere["verts"] if v.co.z < -eps]
            bmesh.ops.translate(bm, verts=tail_sphere["verts"],
                                vec=mathutils.Vector([0, 0, bone.vector.length - tail_shift]))
            bmesh.ops.delete(bm, geom=to_delete, context='VERTS')

            boundary = [edge for edge in bm.edges if edge.is_boundary]
            bmesh.ops.bridge_loops(bm, edges=boundary)

            bmesh.ops.transform(bm, verts=bm.verts, matrix=mathutils.Matrix.Rotation(-math.pi / 2, 4, 'X'))

            if matrix!=None:
                bmesh.ops.transform(bm, verts=bm.verts, matrix=matrix)

            bm.to_mesh(out_mesh)

            bm.free()

        return out_mesh

    def make_convex(self,mesh):
        if len(mesh.vertices)>=3:
            bm = bmesh.new()
            bm.from_mesh(mesh)
            out=bmesh.ops.convex_hull(bm,input=bm.verts)
            bmesh.ops.delete(bm, geom=out["geom_interior"], context='VERTS')
            # bmesh.ops.delete(bm, geom=out["geom_unused"], context=1)
            # bmesh.ops.delete(bm, geom=out["geom_holes"], context=1)
            bm.to_mesh(mesh)
            bm.free()
        return mesh





    @inner_pgroups_to_pointers
    class object(bpy.types.PropertyGroup):
        @pgroup_from_type(bpy.types.RigidBodyObject, **default_rb_settings)
        class rigid_body(bpy.types.PropertyGroup): pass

    settings_tab : bpy.props.EnumProperty(name="Settings",
                                          items=[
                                              ('GENERAL', "General settings", "Settings of generated hitboxes ", 'MESH_DATA', 0),
                                              ('RIGID_BODY', "Rigid body settings", "Rigid body settings of generated hitboxes",'MESH_ICOSPHERE', 1)
                                          ]
                                          )

    use_selected_bones_only : bpy.props.BoolProperty(name="Use selected bones only")
    show_in_front: bpy.props.BoolProperty(name="Make the object draw in front of others")

    generate_from : bpy.props.EnumProperty(name="Generate from",
                                           items=[
                                               ('MESH',"Mesh","Generate hitboxes from vertex groups of the selected mesh",'NONE',0),
                                               ('BONES', "Bones","Generate hitboxes from bones", 'NONE', 1),

                                           ]
                                           )
    hit_box_collection : bpy.props.StringProperty(name="Hitbox collection",default=hit_box_group_name,
                                                  description="Group name of generated hitboxes")

    class mesh_settings(bpy.types.PropertyGroup):
        radius : bpy.props.FloatProperty(name="Radius", default=0.1)
        segments : bpy.props.IntProperty(name="Segments", default=10)

        u_segments : bpy.props.IntProperty()
        v_segments : bpy.props.IntProperty()

        use_envelope : bpy.props.BoolProperty(name="Use bone envelope")

        head_radius : bpy.props.FloatProperty(name="Head radius", min=0, default=0.2)
        tail_radius : bpy.props.FloatProperty(name="Tail radius", min=0, default=0.2)

        envelope_distance_mlt : bpy.props.FloatProperty(name="Distance multiplier", min=0, default=0)
        head_radius_mlt : bpy.props.FloatProperty(name="Head radius multiplier", min=0, default=1)
        tail_radius_mlt : bpy.props.FloatProperty(name="Tail radius multiplier", min=0, default=1)

        head_shift : bpy.props.FloatProperty(name="Head shift", default=0.)
        tail_shift : bpy.props.FloatProperty(name="Tail shift", default=0.)

        scale:bpy.props.FloatVectorProperty(name="Scale",default=(1.,1.,1.),)


        bone_shape: bpy.props.EnumProperty(name="Hitbox shape",
                                           items=[
                                               ('CAPSULE', "Capsule", "Create hitboxes as capsules", 'NONE', 0),
                                               ('BOX', "Box", "Create hitboxes as boxes", 'NONE', 1),
                                           ]
                                           )

        mesh_shape:bpy.props.EnumProperty(name="Mesh type",default='FANCY',
                                          items=[
                                              ('PARTS', "Mesh parts", "Shape from vertex groups (slow)", 'NONE', 0),
                                              ('CONVEX', "Convex", "Convex shapes from vertex groups(very slow)", 'NONE', 1),
                                              ('FANCY', "Fancy", "Smooth shape from vertex groups (slowest) ",'NONE', 2)
                                          ]
                                          )
        use_normalize:bpy.props.BoolProperty(name="Normalize",description="Normalize vertex group weights before processing",default=True)
        weight_threshold:bpy.props.FloatProperty(name="Threshold",description="Result mesh will only include verticies with a weight above this threshold", min=0, default=0.5,max=1.0)
        resolution:bpy.props.IntProperty(name="Mesh resolution", min=0, default=1)

        smooth:bpy.props.FloatProperty(name="Smooth",description="Smooth hitbox", min=0, default=1.0)
        inflate: bpy.props.FloatProperty(name="Inflate",description="Inflate hitbox",default=0.0)



    class rig_settings(bpy.types.PropertyGroup):
        create_rig: bpy.props.BoolProperty(name="Connect hitboxes to bones", default=True)
        empty_display_type: prop_from_type(bpy.types.Object.bl_rna.properties["empty_display_type"],default='CUBE')
        empty_display_size: prop_from_type(bpy.types.Object.bl_rna.properties["empty_display_size"],default=0.2)
        helpers_group: bpy.props.StringProperty(name="Group", default=helpers_group_name,description="Collection name of helper empties")

    show_rb_settings : bpy.props.BoolProperty(name="Show rigid body settings")

    # def check(self, context):
    #     return False



    def draw(self, context):
        # print("DRAW")
        layout = self.layout

        layout.row().prop(self, "settings_tab", expand=False,text="")
        layout.separator()

        if self.settings_tab == 'GENERAL':
            layout.prop(self,"use_selected_bones_only")
            layout.prop(self, "generate_from",text="Make from")

            # layout.separator()
            # layout.label("Mesh settings:")
            layout.separator()

            if self.generate_from=='BONES':

                layout.prop(self.mesh_settings, "bone_shape",text="Shape")


                if self.mesh_settings.bone_shape == 'CAPSULE':
                    layout.prop(self.mesh_settings, "segments")

                sub = layout.column(align=True)
                sub.prop(self.mesh_settings, "head_shift")
                sub.prop(self.mesh_settings, "tail_shift")
                sub.label(text="Scale:")
                sub.row().prop(self.mesh_settings, "scale",text="")
                layout.prop(self.mesh_settings, "use_envelope")
                sub = layout.column(align=True)
                if self.mesh_settings.use_envelope:
                    sub.prop(self.mesh_settings, "head_radius_mlt")
                    sub.prop(self.mesh_settings, "tail_radius_mlt")
                    layout.prop(self.mesh_settings, "envelope_distance_mlt")
                else:
                    sub.prop(self.mesh_settings, "head_radius")
                    sub.prop(self.mesh_settings, "tail_radius")

            if self.generate_from== 'MESH':
                layout.prop(self.mesh_settings, "mesh_shape",text="Shape")
                layout.prop(self.mesh_settings, "use_normalize")
                layout.prop(self.mesh_settings, "weight_threshold")
                if self.mesh_settings.mesh_shape== 'FANCY':
                    layout.prop(self.mesh_settings, "resolution")
                    layout.prop(self.mesh_settings,"smooth")
                    layout.prop(self.mesh_settings,"inflate")
            layout.separator()
            layout.prop(self, "hit_box_collection",text="Collection")
            layout.prop(self,"show_in_front",text="In Front")


        if self.settings_tab == 'RIGID_BODY':
            rb_panel(layout,self)

    # def invoke(self, context, event):
    #     # print("INVOKE")
    #     gv.disable_handler=True
    #     return {'FINISHED'}

    def execute(self, context):
        # print("EXECUTE BEGIN")
        context.view_layer.update()

        update_pairs()

        pose_bone_list = []
        mesh_list=[]

        if context.mode == 'POSE':
            if self.use_selected_bones_only:
                pose_bone_list = [pose_bone for pose_bone in context.selected_pose_bones if None in get_pair(pose_bone)]
            else:
                pose_bone_list = [pose_bone for pose_bone in context.visible_pose_bones if None in get_pair(pose_bone)]

            if self.generate_from == 'MESH':
                mesh_list=[selected_object for selected_object in  context.selected_objects if is_mesh(selected_object) ]




        else:
            for selected_object in context.selected_objects:
                if is_armature(selected_object):
                    for pose_bone in selected_object.pose.bones:
                        if not pose_bone.bone.hide:
                            if pose_bone.bone.select or not self.use_selected_bones_only:
                                if any([bl and al for bl, al in zip(pose_bone.bone.layers, selected_object.data.layers)]):
                                    if None in get_pair(pose_bone):
                                        pose_bone_list.append(pose_bone)
                if self.generate_from== 'MESH':
                    if is_mesh(selected_object):
                        mesh_list.append(selected_object)




        to_rigid = []



        if self.generate_from== 'MESH':
            if len(mesh_list)==1:
                remove_rig(pose_bone_list)
                graph=context.evaluated_depsgraph_get()

                source_object=mesh_list[0].copy()

                base_meshes=[]
                face_counters=[]

                meshed_pose_bones=[]


                if self.mesh_settings.use_normalize:
                    over_context=context.copy()
                    over_context["object"]=source_object
                    bpy.ops.object.vertex_group_normalize_all(over_context,group_select_mode='ALL', lock_active=False)
                for pose_bone in pose_bone_list:

                    if pose_bone.name in source_object.vertex_groups:
                        vertex_group=source_object.vertex_groups[pose_bone.name]
                        base_mesh=source_object.copy()




                        weight_edit = base_mesh.modifiers.new("", 'VERTEX_WEIGHT_EDIT')
                        weight_edit.use_remove = True
                        weight_edit.remove_threshold = self.mesh_settings.weight_threshold
                        weight_edit.vertex_group = vertex_group.name

                        mask = base_mesh.modifiers.new("", 'MASK')
                        mask.vertex_group = vertex_group.name


                        decimate=base_mesh.modifiers.new("", 'DECIMATE')
                        face_counters.append(decimate)


                        bpy.context.collection.objects.link(base_mesh)

                        base_meshes.append(base_mesh)
                        meshed_pose_bones.append(pose_bone)



                context.view_layer.update()
                graph=context.evaluated_depsgraph_get()


                if self.mesh_settings.mesh_shape=='FANCY':
                    to_remove=[]

                    smooth_factor=1.79
                    fancy_sources=[]
                    first_wrappers=[]
                    second_wrappers=[]
                    wrapped_bones=[]

                    for base_mesh,face_counter,pose_bone in zip(base_meshes,face_counters,meshed_pose_bones):
                        base_mesh.modifiers.update()
                        if face_counter.face_count>=1 and 0 not in base_mesh.dimensions:
                            evaluated_mesh=base_mesh.evaluated_get(graph)
                            fancy_source=bpy.data.objects.new(self.get_hit_box_name(pose_bone),bpy.data.meshes.new_from_object(evaluated_mesh))
                            fancy_source.data.materials.clear()
                            fancy_source.data.name=self.get_hit_box_name(pose_bone)
                            fancy_source.show_in_front=self.show_in_front
                            fancy_source.matrix_world=base_mesh.matrix_world.copy()
                            set_origin(fancy_source,compose_matrix(fancy_source.matrix_world@get_center_of_mass(fancy_source.data),pose_bone.matrix_world.to_quaternion()))
                            fancy_source.data.update()

                            bounding_box_mesh=self.bounding_box_mesh(fancy_source)

                            if bounding_box_mesh is not None:

                                bpy.context.collection.objects.link(fancy_source)

                                first_wrapper=fancy_source.copy()
                                first_wrapper.data=bounding_box_mesh


                                subdivide=first_wrapper.modifiers.new("",'SUBSURF')
                                subdivide.levels=self.mesh_settings.resolution


                                second_wrapper=first_wrapper.copy()


                                displace=first_wrapper.modifiers.new("",'DISPLACE')
                                displace.strength=smooth_factor*sum(first_wrapper.dimensions)/3


                                shrink_wrap=first_wrapper.modifiers.new("",'SHRINKWRAP')
                                shrink_wrap.target=fancy_source

                                bpy.context.collection.objects.link(first_wrapper)
                                bpy.context.collection.objects.link(second_wrapper)

                                fancy_sources.append(fancy_source)
                                first_wrappers.append(first_wrapper)
                                second_wrappers.append(second_wrapper)
                                wrapped_bones.append(pose_bone)

                            else:
                                to_remove.append(fancy_source)

                    context.view_layer.update()
                    graph=context.evaluated_depsgraph_get()



                    for fancy_source,first_wrapper,second_wrapper,wrapped_bone in zip(fancy_sources,first_wrappers,second_wrappers,wrapped_bones):


                        second_wrapper.data=bpy.data.meshes.new_from_object(second_wrapper.evaluated_get(graph))
                        second_wrapper.modifiers.clear()

                        displace=second_wrapper.modifiers.new("",'DISPLACE')
                        displace.strength=smooth_factor*sum(first_wrapper.dimensions)/3

                        shrink_wrap=second_wrapper.modifiers.new("",'SHRINKWRAP')
                        shrink_wrap.target=first_wrapper


                        smooth=second_wrapper.modifiers.new("",'CORRECTIVE_SMOOTH')
                        if self.mesh_settings.smooth==0:
                            smooth.factor=0
                            smooth.iterations=0
                        else:
                            smooth.factor=self.mesh_settings.smooth/math.ceil(self.mesh_settings.smooth)
                            smooth.iterations=math.ceil(self.mesh_settings.smooth)
                        smooth.smooth_type='LENGTH_WEIGHTED'


                        inflate=second_wrapper.modifiers.new("",'DISPLACE')
                        inflate.strength=self.mesh_settings.inflate

                        to_remove.append(fancy_source)
                        to_remove.append(first_wrapper)


                    context.view_layer.update()
                    graph=context.evaluated_depsgraph_get()

                    for second_wrapper in second_wrappers:
                        mesh=second_wrapper.data
                        second_wrapper.data=bpy.data.meshes.new_from_object(second_wrapper.evaluated_get(graph))
                        second_wrapper.modifiers.clear()
                        bpy.data.meshes.remove(mesh,do_unlink=True)



                    for mesh_object in to_remove:
                        mesh=mesh_object.data
                        bpy.data.objects.remove(mesh_object, do_unlink=True)
                        bpy.data.meshes.remove(mesh, do_unlink=True)

                    for base_mesh in base_meshes:
                        bpy.data.objects.remove(base_mesh,do_unlink=True)



                    for out_mesh,pose_bone in zip(second_wrappers,wrapped_bones):
                        out_mesh.name=self.get_hit_box_name(pose_bone)
                        out_mesh.data.name=out_mesh.name
                        link_hit_box_and_bone(out_mesh,pose_bone,context)
                        to_rigid.append(out_mesh)

                else:
                    for base_mesh, face_counter,pose_bone in zip(base_meshes, face_counters,meshed_pose_bones):
                        base_mesh.modifiers.update()
                        # print()
                        # print(base_mesh,face_counter.face_count)
                        # print(base_mesh.is_evaluated,base_mesh.users_collection)

                        if face_counter.face_count >= 1:
                            evaluated_mesh=base_mesh.evaluated_get(graph)
                            if self.mesh_settings.mesh_shape == 'PARTS':
                                out_mesh = bpy.data.objects.new(self.get_hit_box_name(pose_bone),bpy.data.meshes.new_from_object(evaluated_mesh))
                            if self.mesh_settings.mesh_shape == 'CONVEX':
                                out_mesh=bpy.data.objects.new(self.get_hit_box_name(pose_bone),self.make_convex(bpy.data.meshes.new_from_object(evaluated_mesh)))

                            out_mesh.data.materials.clear()
                            out_mesh.data.name = self.get_hit_box_name(pose_bone)
                            out_mesh.show_in_front = self.show_in_front
                            out_mesh.matrix_world=base_mesh.matrix_world.copy()
                            set_origin(out_mesh,compose_matrix(out_mesh.matrix_world@get_center_of_mass(out_mesh.data),pose_bone.matrix_world.to_quaternion()))
                            out_mesh.data.update()
                            link_hit_box_and_bone(out_mesh,pose_bone,context)
                            to_rigid.append(out_mesh)

                            bpy.data.objects.remove(base_mesh, do_unlink=True)


                bpy.data.objects.remove(source_object, do_unlink=True)


        else:
            remove_rig(pose_bone_list)
            for pose_bone in pose_bone_list:
                mesh = None
                matrix=mathutils.Matrix.Scale(self.mesh_settings.scale[0],4,(1,0,0))@mathutils.Matrix.Scale(self.mesh_settings.scale[1],4,(0,1,0))@mathutils.Matrix.Scale(self.mesh_settings.scale[2],4,(0,0,1))
                if self.mesh_settings.bone_shape == 'CAPSULE':
                    mesh = self.capsule_mesh(pose_bone.bone,matrix)
                if self.mesh_settings.bone_shape == 'BOX':
                    mesh = self.box_mesh(pose_bone.bone,matrix)

                if mesh != None:
                    hit_box = self.make_hit_box(mesh, pose_bone, context)
                    hit_box.show_in_front=self.show_in_front
                    to_rigid.append(hit_box)
                    if self.rig_settings.create_rig:
                        link_hit_box_and_bone(hit_box,pose_bone,context)
                        pass
                    # bpy.ops.wm.redraw_timer(type='DRAW_WIN_SWAP', iterations=1)

        if len(to_rigid) != 0:
            if self.hit_box_collection in bpy.data.collections:
                collection = bpy.data.collections[self.hit_box_collection]
            else:
                collection = bpy.data.collections.new(self.hit_box_collection)
                context.scene.collection.children.link(collection)
            for item in to_rigid:
                if item.name in context.scene.collection.objects:
                    context.scene.collection.objects.unlink(item)

                collection.objects.link(item)

        make_rigid(to_rigid, context, self.object.rigid_body)

        # print("EXECUTE END")



        return {'FINISHED'}


def rb_panel(layout,context):

    layout.use_property_split=True

    ob=context.object
    rbo=ob.rigid_body


    layout.prop(rbo,"type",text="Type")

    # ---------------

    layout.use_property_split=True



    flow=layout.grid_flow(row_major=True,columns=0,even_columns=True,even_rows=False,align=True)
    col=flow.column()

    if rbo.type=='ACTIVE':
        col.prop(rbo,"mass")
        col.prop(rbo,"enabled",text="Dynamic")

    col=flow.column()
    col.prop(rbo,"kinematic",text="Animated")


    # ---------------

    layout.prop(rbo,"collision_shape",text="Shape")

    if rbo.collision_shape in {'MESH','CONVEX_HULL'}:
        layout.prop(rbo,"mesh_source",text="Source")

    if rbo.collision_shape=='MESH' and rbo.mesh_source=='DEFORM':
        layout.prop(rbo,"use_deform",text="Deforming")

    # ---------------
    flow=layout.grid_flow(row_major=True,columns=0,even_columns=True,even_rows=False,align=True)


    col=flow.column()
    col.prop(rbo,"friction")

    col=flow.column()
    col.prop(rbo,"restitution",text="Bounciness")

    # ---------------

    if rbo.collision_shape in {'MESH','CONE'}:
        col=layout.column()
        col.prop(rbo,"collision_margin",text="Margin")
    else:
        flow=layout.grid_flow(row_major=True,columns=0,even_columns=True,even_rows=False,align=True)
        col=flow.column()
        col.prop(rbo,"use_margin")

        col=flow.column()
        col.active=rbo.use_margin
        col.prop(rbo,"collision_margin",text="Margin")

    # ---------------

    col=layout.column()
    col.ui_units_y=2

    col.prop(rbo,"collision_collections",text="")

    # ---------------

    flow=layout.grid_flow(row_major=True,columns=0,even_columns=True,even_rows=False,align=True)


    col=flow.column()
    col.prop(rbo,"linear_damping",text="Damping Translation")

    col=flow.column()
    col.prop(rbo,"angular_damping",text="Rotation")

def rbc_panel(layout,context):

    ob=context.object
    rbc=ob.rigid_body_constraint

    layout.prop(rbc,"type")

    row=layout.row()
    row.prop(rbc,"enabled")
    row.prop(rbc,"disable_collisions")


    if rbc.type!='MOTOR':
        row=layout.row()
        row.prop(rbc,"use_breaking")
        sub=row.row()
        sub.active=rbc.use_breaking
        sub.prop(rbc,"breaking_threshold",text="Threshold")

    row=layout.row()
    row.prop(rbc,"use_override_solver_iterations",text="Override Iterations")
    sub=row.row()
    sub.active=rbc.use_override_solver_iterations
    sub.prop(rbc,"solver_iterations",text="Iterations")

    if rbc.type=='HINGE':
        col=layout.column(align=True)
        col.label(text="Limits:")

        row=col.row(align=True)
        sub=row.row(align=True)
        sub.scale_x=0.5
        sub.prop(rbc,"use_limit_ang_z",toggle=True)
        sub=row.row(align=True)
        sub.active=rbc.use_limit_ang_z
        sub.prop(rbc,"limit_ang_z_lower",text="Lower")
        sub.prop(rbc,"limit_ang_z_upper",text="Upper")

    elif rbc.type=='SLIDER':
        col=layout.column(align=True)
        col.label(text="Limits:")

        row=col.row(align=True)
        sub=row.row(align=True)
        sub.scale_x=0.5
        sub.prop(rbc,"use_limit_lin_x",toggle=True)
        sub=row.row(align=True)
        sub.active=rbc.use_limit_lin_x
        sub.prop(rbc,"limit_lin_x_lower",text="Lower")
        sub.prop(rbc,"limit_lin_x_upper",text="Upper")

    elif rbc.type=='PISTON':
        col=layout.column(align=True)
        col.label(text="Limits:")

        row=col.row(align=True)
        sub=row.row(align=True)
        sub.scale_x=0.5
        sub.prop(rbc,"use_limit_lin_x",toggle=True)
        sub=row.row(align=True)
        sub.active=rbc.use_limit_lin_x
        sub.prop(rbc,"limit_lin_x_lower",text="Lower")
        sub.prop(rbc,"limit_lin_x_upper",text="Upper")

        col=layout.column(align=True)

        row=col.row(align=True)
        sub=row.row(align=True)
        sub.scale_x=0.5
        sub.prop(rbc,"use_limit_ang_x",toggle=True)
        sub=row.row(align=True)
        sub.active=rbc.use_limit_ang_x
        sub.prop(rbc,"limit_ang_x_lower",text="Lower")
        sub.prop(rbc,"limit_ang_x_upper",text="Upper")

    elif rbc.type=='MOTOR':
        col=layout.column(align=True)
        col.label(text="Linear motor:")

        row=col.row(align=True)
        sub=row.row(align=True)
        sub.scale_x=0.5
        sub.prop(rbc,"use_motor_lin",toggle=True,text="Enable")
        sub=row.row(align=True)
        sub.active=rbc.use_motor_lin
        sub.prop(rbc,"motor_lin_target_velocity",text="Target Velocity")
        sub.prop(rbc,"motor_lin_max_impulse",text="Max Impulse")

        col.label(text="Angular motor:")

        row=col.row(align=True)
        sub=row.row(align=True)
        sub.scale_x=0.5
        sub.prop(rbc,"use_motor_ang",toggle=True,text="Enable")
        sub=row.row(align=True)
        sub.active=rbc.use_motor_ang
        sub.prop(rbc,"motor_ang_target_velocity",text="Target Velocity")
        sub.prop(rbc,"motor_ang_max_impulse",text="Max Impulse")

    elif rbc.type in {'GENERIC','GENERIC_SPRING'}:
        col=layout.column(align=True)
        col.label(text="Limits:")

        row=col.row(align=True)
        sub=row.row(align=True)
        sub.scale_x=0.5
        sub.prop(rbc,"use_limit_lin_x",toggle=True)
        sub=row.row(align=True)
        sub.active=rbc.use_limit_lin_x
        sub.prop(rbc,"limit_lin_x_lower",text="Lower")
        sub.prop(rbc,"limit_lin_x_upper",text="Upper")

        row=col.row(align=True)
        sub=row.row(align=True)
        sub.scale_x=0.5
        sub.prop(rbc,"use_limit_lin_y",toggle=True)
        sub=row.row(align=True)
        sub.active=rbc.use_limit_lin_y
        sub.prop(rbc,"limit_lin_y_lower",text="Lower")
        sub.prop(rbc,"limit_lin_y_upper",text="Upper")

        row=col.row(align=True)
        sub=row.row(align=True)
        sub.scale_x=0.5
        sub.prop(rbc,"use_limit_lin_z",toggle=True)
        sub=row.row(align=True)
        sub.active=rbc.use_limit_lin_z
        sub.prop(rbc,"limit_lin_z_lower",text="Lower")
        sub.prop(rbc,"limit_lin_z_upper",text="Upper")

        col=layout.column(align=True)

        row=col.row(align=True)
        sub=row.row(align=True)
        sub.scale_x=0.5
        sub.prop(rbc,"use_limit_ang_x",toggle=True)
        sub=row.row(align=True)
        sub.active=rbc.use_limit_ang_x
        sub.prop(rbc,"limit_ang_x_lower",text="Lower")
        sub.prop(rbc,"limit_ang_x_upper",text="Upper")

        row=col.row(align=True)
        sub=row.row(align=True)
        sub.scale_x=0.5
        sub.prop(rbc,"use_limit_ang_y",toggle=True)
        sub=row.row(align=True)
        sub.active=rbc.use_limit_ang_y
        sub.prop(rbc,"limit_ang_y_lower",text="Lower")
        sub.prop(rbc,"limit_ang_y_upper",text="Upper")

        row=col.row(align=True)
        sub=row.row(align=True)
        sub.scale_x=0.5
        sub.prop(rbc,"use_limit_ang_z",toggle=True)
        sub=row.row(align=True)
        sub.active=rbc.use_limit_ang_z
        sub.prop(rbc,"limit_ang_z_lower",text="Lower")
        sub.prop(rbc,"limit_ang_z_upper",text="Upper")

        if rbc.type=='GENERIC_SPRING':
            col=layout.column(align=True)
            col.label(text="Springs:")

            row=col.row(align=True)
            sub=row.row(align=True)
            sub.scale_x=0.5
            sub.prop(rbc,"use_spring_x",toggle=True,text="X Axis")
            sub=row.row(align=True)
            sub.active=rbc.use_spring_x
            sub.prop(rbc,"spring_stiffness_x",text="Stiffness")
            sub.prop(rbc,"spring_damping_x",text="Damping")

            row=col.row(align=True)
            sub=row.row(align=True)
            sub.scale_x=0.5
            sub.prop(rbc,"use_spring_y",toggle=True,text="Y Axis")
            sub=row.row(align=True)
            sub.active=rbc.use_spring_y
            sub.prop(rbc,"spring_stiffness_y",text="Stiffness")
            sub.prop(rbc,"spring_damping_y",text="Damping")

            row=col.row(align=True)
            sub=row.row(align=True)
            sub.scale_x=0.5
            sub.prop(rbc,"use_spring_z",toggle=True,text="Z Axis")
            sub=row.row(align=True)
            sub.active=rbc.use_spring_z
            sub.prop(rbc,"spring_stiffness_z",text="Stiffness")
            sub.prop(rbc,"spring_damping_z",text="Damping")

            col=layout.column(align=True)

            row=col.row(align=True)
            sub=row.row(align=True)
            sub.scale_x=0.5
            sub.prop(rbc,"use_spring_ang_x",toggle=True,text="X Angle")
            sub=row.row(align=True)
            sub.active=rbc.use_spring_ang_x
            sub.prop(rbc,"spring_stiffness_ang_x",text="Stiffness")
            sub.prop(rbc,"spring_damping_ang_x",text="Damping")

            row=col.row(align=True)
            sub=row.row(align=True)
            sub.scale_x=0.5
            sub.prop(rbc,"use_spring_ang_y",toggle=True,text="Y Angle")
            sub=row.row(align=True)
            sub.active=rbc.use_spring_ang_y
            sub.prop(rbc,"spring_stiffness_ang_y",text="Stiffness")
            sub.prop(rbc,"spring_damping_ang_y",text="Damping")

            row=col.row(align=True)
            sub=row.row(align=True)
            sub.scale_x=0.5
            sub.prop(rbc,"use_spring_ang_z",toggle=True,text="Z Angle")
            sub=row.row(align=True)
            sub.active=rbc.use_spring_ang_z
            sub.prop(rbc,"spring_stiffness_ang_z",text="Stiffness")
            sub.prop(rbc,"spring_damping_ang_z",text="Damping")



class RAGDOLLTOOLS_MT_RagdollPieMenu(bpy.types.Menu):
    """I am help string"""
    bl_label = "Ragdoll pie menu"

    def draw(self, context):
        layout = self.layout

        separation_factor=0.3

        pie = layout.menu_pie()



        column=pie.column(align=True)
        column.scale_y=1.5
        inner_column=column.column(align=True)
        inner_column.scale_y=1.5
        inner_column.operator(GenerateHitBoxes.bl_idname,text="Make hitboxes",icon='MESH_CAPSULE')

        column.separator(factor=separation_factor)

        column.operator(MakeVertexGroups.bl_idname,text="Make vertex groups",icon='GROUP_VERTEX')
        column.operator(ClearVertexGroups.bl_idname,text="Clear vertex groups",icon='X')

        column.separator(factor=separation_factor)

        column.operator(LinkHitBoxAndBone.bl_idname,icon='LINKED')
        column.operator(UnlinkHitBoxAndBone.bl_idname,icon='UNLINKED')

        column.separator(factor=separation_factor)
        column.operator(SetMass.bl_idname,text="Set mass",icon='MOD_VERTEX_WEIGHT')
        column.separator(factor=separation_factor)
        column.operator(SetRBSettings.bl_idname,text="Copy rigid body settings",icon='COPYDOWN')











        column=pie.column(align=True)

        row=column.row(align=True)
        row.scale_x=row.scale_y=1.5

        inner_column=row.column(align=True)
        inner_column.scale_x=1.0
        inner_column.operator(KeyframerDisable.bl_idname,text="Dynamic",icon='PHYSICS')
        inner_column.operator(KeyframerEnable.bl_idname,text="Animated",icon='ARMATURE_DATA')

        inner_column=row.column(align=True)
        inner_column.scale_y=2
        inner_column.scale_x=1
        row=inner_column.row(align=True)
        operator=row.operator(KeyframerInsert.bl_idname,text="",icon='KEY_HLT')
        # operator.mode='INSERT'
        # operator.target='IS_PINNED'

        operator=row.operator(KeyframerDelete.bl_idname,text="",icon='KEY_DEHLT')
        # operator.mode='DELETE'
        # operator.target='IS_PINNED'

        operator=row.operator(KeyframerClear.bl_idname,text="",icon='X')
        # operator.mode='CLEAR'
        # operator.target='IS_PINNED'




        column.separator(factor=separation_factor)



        inner_column=column.column(align=True)
        inner_column.scale_y=1.5


        inner_column.operator(FollowHitbox.bl_idname,text="Follow hitboxes",icon='MESH_CAPSULE')
        inner_column.operator(FollowBone.bl_idname,text="Follow bones",icon='BONE_DATA')


        inner_column.separator(factor=separation_factor)


        inner_column.operator(BakeToBone.bl_idname,text="Bake",icon='KEYINGSET')
        inner_column.separator(factor=separation_factor)
        inner_column.operator(ClearCache.bl_idname,text="Clear cache",icon='X')












        column=pie.column(align=True)
        # column.scale_x=1.5
        # column.scale_y=1.5
        # column.operator(SetMass.bl_idname,text="Set mass",icon='MOD_VERTEX_WEIGHT')
        # column.operator(SetRBSettings.bl_idname,text="Copy rigid body settings",icon='COPYDOWN')



        column=pie.column(align=True)
        column.scale_y=1.5
        inner_column=column.column(align=True)
        inner_column.scale_y=1.5

        inner_column.operator(MakeRigidBodyConstraints.bl_idname,text="Make constraints",icon='RIGID_BODY_CONSTRAINT')
        column.separator(factor=separation_factor)
        column.operator(SetLimits.bl_idname,text="Set limits",icon='DRIVER_ROTATIONAL_DIFFERENCE').mode='START'
        column.separator(factor=separation_factor)
        column.operator(Snap.bl_idname,icon='SNAP_ON')
        column.separator(factor=separation_factor)
        column.operator(SetRBCSettings.bl_idname,text="Copy constraint settings",icon='COPYDOWN')







class RagdollPieMenuStarter(bpy.types.Operator):
    """Ragdoll tools pie menu"""
    bl_idname = "ragdoll.pie_menu_start"
    bl_label = "Ragdoll tools"
    bl_options={'REGISTER'}

    def invoke(self, context, event):
        bpy.ops.wm.call_menu_pie(name="RAGDOLLTOOLS_MT_RagdollPieMenu")
        return {'FINISHED'}


register_classes = [BakeToBone,LinkHitBoxAndBone,UnlinkHitBoxAndBone,SetRBCSettings,SetRBSettings,MakeRigidBodyConstraints,GenerateHitBoxes,RAGDOLLTOOLS_MT_RagdollPieMenu,RagdollPieMenuStarter,Snap,SetMass,SetLimits,MakeVertexGroups,ClearVertexGroups,ClearCache]
register_classes.extend([Keyframer,KeyframerInsert,KeyframerDelete,KeyframerClear,KeyframerEnable,KeyframerDisable])
register_classes.extend([SwitchSource,FollowBone,FollowHitbox])


def draw_starter(self,context):
    pass
    self.layout.operator(RagdollPieMenuStarter.bl_idname)

def register():
    # os.system('cls')
    # print("I AM REGISTER")

    register_extensions()


    for register_class in register_classes:
        bpy.utils.register_class(register_class)


    bpy.types.VIEW3D_MT_view.append(draw_starter)



    # print("REGISTER FINISHED")


def unregister():
    # print("I AM UNREGISTER")
    for register_class in reversed(register_classes):
        bpy.utils.unregister_class(register_class)

    bpy.types.VIEW3D_MT_view.remove(draw_starter)



    # print("UNREGISTER FINISHED")


